#pragma once

#include <mail/unistat/cpp/include/meters/common.h>

#include <boost/accumulators/accumulators.hpp>
#include <boost/accumulators/statistics.hpp>
#include <boost/range/adaptors.hpp>
#include <re2/re2.h>

namespace unistat {

struct CountErrorCategories {
    explicit CountErrorCategories(std::unordered_map<std::string, std::string> expectedErrors, std::string namePrefix)
            : _namePrefix(std::move(namePrefix))
            , _nameMapping(std::move(expectedErrors))
            , _unexpected(0)
            , _errors()
    {
        for (auto& e: _nameMapping | boost::adaptors::map_values) {
            _errors[e] = 0;
        }
    }

    void update(const std::map<std::string, std::string>& record) {
        const auto level = record.find("level");
        if (record.end() == level || "error" != level->second) {
            return;
        }

        const auto ecValue = record.find("error_code.category");
        if (record.end() != ecValue) {
            const auto category = _nameMapping.find(ecValue->second);
            if (_nameMapping.end() == category) {
                _unexpected += 1;
            } else {
                _errors[category->second] += 1;
            }
        }
    }

    std::vector<NamedValue<std::size_t>> get() const {
        const auto metrics = _errors | boost::adaptors::transformed([&prefix = _namePrefix](const auto& e){
            const std::string name = normalizeName(buildSignalName(prefix, e.first));
            return NamedValue<std::size_t>{withSigoptSuffix(name, "summ"), e.second};
        });
        std::vector<NamedValue<std::size_t>> result {metrics.begin(), metrics.end()};

        const std::string unexpectedName = normalizeName(buildSignalName(_namePrefix, "unexpected_error"));
        result.emplace_back(withSigoptSuffix(unexpectedName, "summ"), _unexpected);

        return result;
    }

private:
    std::string _namePrefix;
    std::unordered_map<std::string, std::string> _nameMapping;
    std::size_t _unexpected;
    std::unordered_map<std::string, std::size_t> _errors;
};

} // namespace unistat
