from argparse import ArgumentParser
from collections import namedtuple
from mail.unistat.cpp.cython.meters import (
    HttpClientHttpRequestTotalTimeHist,
    HttpClientHttpRequestCountByStatus,
    SupervisorLogRestartMeters,
)
from mail.york.unistat.cpp.run import (
    CountErrorCategories,
)
import os
import yaml
import logging
import mail.york.unistat.cpp.run as york_unistat
import mail.unistat.cpp.cython.logs as logs


logging.basicConfig(level=logging.WARNING, format='[%(asctime)s] [%(levelname)s]: %(message)s')
log = logging.getLogger(__name__)


YORK_LOG_ERRORS = {
    'apq.pq': 'apq',
    'macs::error::Category': 'macs',
    'pgg::error::CommonCategory': 'pg_common',
    'pgg::error::ReadonlyCategory': 'pg_readonly',
    'pgg::error::ConnectionCategory': 'pg_connection',
    'pgg::error::SqlCategory': 'pg_sql',
    'pgg::errc::DatabaseCategory': 'pg_database',
    'pgg::query::repository::error': 'pg_repository',
    'sharpei_client': 'sharpei_client',
}


def make_supervisor_meters():
    return [
        SupervisorLogRestartMeters('supervisor')
    ]


def make_york_log_meters(prefix):
    return [
        CountErrorCategories(YORK_LOG_ERRORS, prefix),
    ]


def make_http_client_log_meters():
    return [
        HttpClientHttpRequestCountByStatus("conninfo", "count_by_status_sharpei_conninfo"),
        HttpClientHttpRequestTotalTimeHist((0, 20, 100, 300, 1000), "conninfo", "sharpei_conninfo_time_hist"),
        HttpClientHttpRequestCountByStatus("stat", "count_by_status_sharpei_stat"),
        HttpClientHttpRequestTotalTimeHist((0, 20, 100, 300, 1000), "stat", "sharpei_stat_time_hist"),
    ]


def parse_args():
    parser = ArgumentParser()
    parser.add_argument('-H', '--host', default='::')
    parser.add_argument('-p', '--port', default=8082, type=int)
    parser.add_argument('-d', '--dir', default='.')
    parser.add_argument('-s', action='store_true', help='read file logs from start')
    parser.add_argument('-l', '--log', default='/var/log/york/unistat.log', help='path for yplatform.log')
    parser.add_argument('--supervisorlog', help='path for supervisord.log')
    parser.add_argument('york_config')
    return parser.parse_args()


YorkConfig = namedtuple('YorkConfig', (
    'httpclient_log',
    'york_log',
    'worker_log',
))


def make_york_config(data):
    return YorkConfig(
        httpclient_log=os.path.join(os.curdir, data['config']['log']['http_client']['sinks'][0]['path']),
        york_log=os.path.join(os.curdir, data['config']['log']['york']['sinks'][0]['path']),
        worker_log=os.path.join(os.curdir, data['config']['log']['worker']['sinks'][0]['path']),
    )


def main():
    args = parse_args()
    log.info('chdir %s' % os.path.abspath(args.dir))
    os.chdir(args.dir)

    with open(args.york_config) as f:
        york_config = make_york_config(yaml.load(f, Loader=yaml.FullLoader))

    fast_forward = args.s

    york_log = york_unistat.YorkLog([], make_york_log_meters('york_errors'), fast_forward, york_config.york_log)
    worker_log = york_unistat.YorkLog([], make_york_log_meters('worker_errors'), fast_forward, york_config.worker_log)
    http_client_log = logs.HttpClientLog([], make_http_client_log_meters(), fast_forward, york_config.httpclient_log)

    logs_list = [york_log, worker_log, http_client_log]

    if args.supervisorlog is not None:
        supervisor_log = logs.SupervisorLog(
            [],
            make_supervisor_meters(),
            fast_forward,
            args.supervisorlog
        )
    logs_list.append(supervisor_log)

    york_unistat.run(args.host, args.port, logs_list, args.log, logLevel='info')


if __name__ == '__main__':
    main()
