## Поиск и использование ресурсов

### Реакторы

В конфиге модуля указывается основной реактор. Этот реактор передается параметром в конструктор модуля либо в метод `init()`.

Если основной реактор в конфиге модуля не указан, то по умолчанию это будет "global".

Модуль может искать и использовать другие реакторы, если знает их имена. Найти реактор можно несколькими способами:

1. глобальным поиском
```c++
#include <yplatform/find.h>
auto reactor = yplatform::find_reactor("name");
```
2. поиском в `reactor_set`, который вместо реактора можно принимать в конструкторе модуля
```c++
module(yplatform::reactor_set&, const yplatform::ptree& configuration);
```

### Логгеры

Бывают текстовые и типизированные.

Каждый модуль имеет свой текстовый логгер, который можно получить через метод базового класса `module::logger()`. Загрузчик устанавливает логгеру модуля префикс - имя модуля. Таким образом, записи в текстовом логе всегда размечены именами модулей, которые записали то или иное сообщение.

Если логи выключены либо логгер не сконфигурирован, то все записи в него будут игнорироваться. TSKV логгеры загрузчиком не прокидываются.

Запись в лог делается через специальные макросы, определенные в `yplatform/log.h`.  Подробнее про запись логов можно прочитать в документации к [`yplatform::log`](../log.md).

Как найти логгер:

1. глобальным поиском

```c++
#include <yplatform/find.h>
auto logger = yplatform::find_logger<yplatform::log::source>("global");
```

2. через `io_context`

```c++
#include <yplatform/find.h>
auto http_client = yplatform::find<yplatform::log::source>(io, "global");
```

3. с помощью `log::find()`

```c++
#include <yplatform/log.h>
auto http_client = yplatform::log::find(io, "global");
``` 
У функций `yplatform::log::find` имеется опциональный параметр
```cpp
bool fallback_to_global = true
```
который контролирует их поведение в случае, если логгера с именем `name` не существует:
- `fallback_to_global == true` - при наличии глобального логгера, `name == "global"`, будет возвращен он, иначе будет возвращен логгер, который никуда не пишет данные;
- `fallback_to_global == false` - будет выброшено исключение `std::invalid_argument` с описанием того, что запрошенного логгера не существует.


Для TSKV логгеров аналогично.

Примеры простого кода для записи в лог от имени модуля:

```c++
LINFO_(this) << "event handled";
LINFO_(this, context) << "task completed";
```

### Другие модули

Найти другой модуль можно несколькими способами:

1. глобальным поиском
```c++
#include <ymod_httpclient/call.h>
#include <yplatform/find.h>
auto http_client = yplatform::find<yhttp::call>("http_client");
```
2. через `io_context`
```c++
#include <ymod_httpclient/call.h>
#include <yplatform/find.h>
auto http_client = yplatform::find<yhttp::call>(io, "http_client");
```

`find()` возвращает `shared_ptr` на модуль. Модули могут сохранять указатели, поэтому избегайте циклических ссылок. Если это все-таки случается так, что модули A и B должны ссылаться друг на друга, то попробуйте следующие рецепты:

- в одной из связей замените `shared_ptr` на `weak_ptr`
- если циклическая зависимость неявная (например, ваш модуль через `shared_from_this` захватывается в лямбду и передается как колбэк), попробуйте использовать `yplatform::weak_bind()` или другое аналогичное решение
- определите, возможно ли не хранить указатель, а вместо этого делать `find()` там, где требуется обращение к другому модулю; оцените влияние этого решения на производительность вашего модуля
- если циклической ссылки по каким-то причинам нельзя избавиться, то в fini() обоих модулей сбрасывайте указатели друг на друга

Без необходимости не захватывайте сырой указатель this в колбэки и лямбды. Захватывайте `yplatform::shared_from(this)` и `yplatform::weak_from(this)`, это серьезно облегчит вам отладку модуля.

Пример хорошего решения:

```c++
httpclient->async_run(request, [this, self=shared_from(this)](auto, auto){ ... });
```

Плохое решение:

```c++
httpclient->async_run(request, [this](auto, auto){ ... });
```

При остановке приложения возможно падение приложения, если ваш модуль будет удален загрузчиком раньше, чем httpclient.

Не существует однозначного рецепта, хранить указатель на другой модуль или каждый раз искать через `find()`.  Выбирайте наиболее подходящий для вашего приложения вариант, исходя из тестируемости, гибкости и простоты написания кода.