#include <yplatform/net/dns/resolver.h>
#include <yplatform/log.h>

#include <boost/date_time.hpp>
#include <boost/program_options.hpp>
#include <boost/thread.hpp>

#include <iostream>
#include <fstream>

#define L_INFO(msg) std::cout << msg << std::endl

typedef yplatform::net::dns::resolver resolver;

template <class Cont>
void handle_resolve_a(
    const boost::system::error_code& ec,
    resolver::iterator_a it,
    const std::string& host,
    Cont cont)
{
    if (ec) L_INFO(host << " A: " << ec.message());
    while (it != resolver::iterator_a())
        L_INFO(host << " A: " << *it++);
    return cont();
}

template <class Cont>
void handle_resolve_aaaa(
    const boost::system::error_code& ec,
    resolver::iterator_aaaa it,
    const std::string& host,
    Cont cont)
{
    if (ec) L_INFO(host << " AAAA: " << ec.message());
    while (it != resolver::iterator_aaaa())
        L_INFO(host << " AAAA: " << *it++);
    return cont();
}

template <class Cont>
void handle_resolve_ptr(
    const boost::system::error_code& ec,
    resolver::iterator_ptr it,
    const std::string& host,
    Cont cont)
{
    if (ec) L_INFO(host << " PTR: " << ec.message());
    while (it != resolver::iterator_ptr())
        L_INFO(host << " PTR: " << *it++);
    return cont();
}

template <class Cont>
void handle_resolve_mx(
    const boost::system::error_code& ec,
    resolver::iterator_mx it,
    const std::string& host,
    Cont cont)
{
    if (ec) L_INFO(host << " MX: " << ec.message());
    while (it != resolver::iterator_mx())
        L_INFO(host << " MX: " << *it++);
    return cont();
}

template <class Cont>
void handle_resolve_txt(
    const boost::system::error_code& ec,
    resolver::iterator_txt it,
    const std::string& host,
    Cont cont)
{
    if (ec) L_INFO(host << " TXT: " << ec.message());
    while (it != resolver::iterator_txt())
        L_INFO(host << " TXT: " << *it++);
    return cont();
}

template <class Cont>
void async_resolve_a(resolver& r, const std::string& host, Cont cont)
{
    r.async_resolve_a(host, boost::bind(handle_resolve_a<Cont>, _1, _2, host, cont));
}

template <class Cont>
void async_resolve_aaaa(resolver& r, const std::string& host, Cont cont)
{
    r.async_resolve_aaaa(host, boost::bind(handle_resolve_aaaa<Cont>, _1, _2, host, cont));
}

template <class Cont>
void async_resolve_ptr(resolver& r, const std::string& host, Cont cont)
{
    r.async_resolve_ptr(host, boost::bind(handle_resolve_ptr<Cont>, _1, _2, host, cont));
}

template <class Cont>
void async_resolve_mx(resolver& r, const std::string& host, Cont cont)
{
    r.async_resolve_mx(host, boost::bind(handle_resolve_mx<Cont>, _1, _2, host, cont));
}

template <class Cont>
void async_resolve_txt(resolver& r, const std::string& host, Cont cont)
{
    r.async_resolve_txt(host, boost::bind(handle_resolve_txt<Cont>, _1, _2, host, cont));
}

int main(int argc, char** argv)
{
    std::string host;
    namespace bpo = boost::program_options;
    bpo::options_description cmd_opt("cmd line options");
    cmd_opt.add_options()("help,h", "produce help message")(
        "host", bpo::value<std::string>(&host), "host to resolve");
    bpo::variables_map vm;
    try
    {
        bpo::store(bpo::command_line_parser(argc, argv).options(cmd_opt).run(), vm);
        bpo::notify(vm);
        if (vm.count("help") || vm.size() == 0)
        {
            std::cout << cmd_opt << std::endl;
            return 1;
        }
    }
    catch (const std::exception& e)
    {
        std::cerr << "bad options: " << e.what() << std::endl;
        return -1;
    }

    boost::asio::io_service ios(1);
    yplatform::net::dns::resolver_service drs(ios);
    resolver r(drs);
    async_resolve_a(r, host, std::bind([]() {}));
    async_resolve_aaaa(r, host, std::bind([]() {}));
    async_resolve_ptr(r, host, std::bind([]() {}));
    async_resolve_mx(r, host, std::bind([]() {}));
    async_resolve_txt(r, host, std::bind([]() {}));

    ios.run();

    return 0;
}
