#include <yplatform/net/dns/resolver.h>
#include <yplatform/reactor.h>
#include <yplatform/log.h>

#include <boost/date_time.hpp>
#include <boost/program_options.hpp>
#include <boost/thread.hpp>

#include <iostream>
#include <fstream>
#include <vector>

//#define L_INFO(msg) L_(info) << msg
#define L_INFO(msg) std::cout << msg << std::endl
//#define L_INFO(msg)

typedef yplatform::net::dns::resolver resolver_t;
typedef std::shared_ptr<resolver_t> resolver_ptr_t;

template <class Cont>
void handle_resolve_a(
    const boost::system::error_code& ec,
    resolver_t::iterator_a it,
    const std::string& host,
    Cont cont)
{
    if (ec) L_INFO("*** handle_resolve_a: " << host << ": " << ec.message());
    //  while (it != resolver_t::iterator_a())
    //    L_INFO(host << " A: " << *it++);
    return cont();
}

template <class Cont>
void handle_resolve_aaaa(
    const boost::system::error_code& ec,
    resolver_t::iterator_aaaa it,
    const std::string& host,
    Cont cont)
{
    if (ec) L_INFO("*** handle_resolve_aaaa: " << host << ": " << ec.message());
    //  while (it != resolver_t::iterator_aaaa())
    //    L_INFO(host << " AAAA: " << *it++);
    return cont();
}

template <class Cont>
void async_resolve_a(resolver_t& r, const std::string& host, Cont cont)
{
    r.async_resolve_a(host, boost::bind(handle_resolve_a<Cont>, _1, _2, host, cont));
}

template <class Cont>
void async_resolve_aaaa(resolver_t& r, const std::string& host, Cont cont)
{
    r.async_resolve_aaaa(host, boost::bind(handle_resolve_aaaa<Cont>, _1, _2, host, cont));
}

// -----------
struct options
{
    int ios = 1;
    int threads = 1;
    int resolvers = 1;
};

int main(int argc, char** argv)
{
    options p;

    namespace bpo = boost::program_options;
    bpo::options_description cmd_opt("cmd line options");
    cmd_opt.add_options()("help", "produce help message")(
        "ioservices", bpo::value<int>(&p.ios)->default_value(1), "thread count")(
        "threads", bpo::value<int>(&p.threads)->default_value(1), "thread count")(
        "resolvers", bpo::value<int>(&p.resolvers)->default_value(1), "resolver count");
    bpo::variables_map vm;
    try
    {
        bpo::store(bpo::command_line_parser(argc, argv).options(cmd_opt).run(), vm);
        bpo::notify(vm);
        if (vm.count("help") || vm.size() == 0)
        {
            std::cout << cmd_opt << std::endl;
            return 1;
        }
    }
    catch (const std::exception& e)
    {
        std::cerr << "bad options: " << e.what() << std::endl;
        return -1;
    }

    std::string host;
    getline(std::cin, host);

    yplatform::reactor reactor;
    reactor.init(0, p.ios, p.threads);

    //  auto io = io_pool.io();
    //
    //  for (int i = 0; i < 100000; ++i) {
    //    resolver_t r(*io) __attribute__ ((used));
    //  }

    std::vector<resolver_ptr_t> resolvers;
    yplatform::net::dns::resolver_service drs(*io_pool.io());

    for (int i = 0; i < p.resolvers; ++i)
    {
        resolvers.push_back(std::make_shared<resolver_t>(drs));
    }

    reactor.run();

    std::atomic<unsigned> total_requests{ resolvers.size() };
    std::atomic<unsigned> finished_requests{ 0 };
    boost::posix_time::ptime tm = boost::posix_time::microsec_clock::local_time();

    auto handler = [&total_requests, &finished_requests, &reactor, tm]() {
        finished_requests++;
        if (finished_requests == total_requests)
        {
            std::cout << finished_requests << " queries; time elapsed: "
                      << boost::posix_time::microsec_clock::local_time() - tm << std::endl;
        }
    };

    for (size_t i = 0; i < resolvers.size(); i++)
    {
        resolver_t& r = *resolvers[i % resolvers.size()];
        async_resolve_aaaa(r, host, std::bind(handler));
    }

    while (true)
    {
        sleep(1);
        std::cout << "** finished: " << finished_requests << "/" << total_requests << std::endl;
        if (finished_requests == total_requests)
        {
            reactor.stop();
            sleep(1);
            break;
        }
    }

    L_INFO("bye");

    return 0;
}
