#include <string>
#include <ymod_tvm/module.h>
#include <yplatform/module.h>
#include <yplatform/application/global_init.h>
#include <yplatform/find.h>
#include <yplatform/loader.h>
#include <yplatform/time_traits.h>
#include <thread>
#include <fstream>

using std::string;
using boost::system::error_code;

string CONFIG =
    R"(
config:
    system:
        libpath: [ 'lib64' ]
    log:
    modules:
        module:
        -   system:
                name: http_client
                library: ymod_httpclient
                factory: yhttp::call_impl
            configuration:
                reactor: global
        -   system:
                name: tvm
                library: ymod_tvm
                factory: ymod_tvm::tvm2::impl
            configuration:
                service_id: 2000197
                target_services:
                -   id: 2000198
                    name: proxy
                tvm_host: tvm-api.yandex.net
                tvm_secret_file: /tmp/test_tvm_secret
                keys_update_interval: 24:00:00
                tickets_update_interval: 01:00:00
                retry_interval: 00:10:00
                reactor: global
)";

class module_using_tvm : public yplatform::module
{
public:
    void init(const yplatform::ptree& conf)
    {
        auto tvm_name = conf.get("tvm", "tvm");
        using yplatform::find;
        tvm2 = find<ymod_tvm::tvm2_module, std::shared_ptr>(tvm_name);

        tvm2->subscribe_service_ticket(
            "proxy",
            [](const ymod_tvm::error_code& error, const string& service, const string& ticket) {
                if (error)
                {
                    YLOG_G(error) << "tvm2 error: " << error.message();
                    return;
                }
                YLOG_G(error) << "tvm2 '" << service << "' ticket: " << ticket;
            });
        periodic_event();
    }

private:
    void periodic_event(const error_code& error = error_code())
    {
        --event_count;
        string ticket;

        if (auto res = tvm2->get_service_ticket("proxy", ticket))
        {
            YLOG_G(error) << "tvm2 getter error: " << res.message();
        }
        else
        {
            YLOG_G(error) << "tvm2 getter 'proxy' ticket: " << ticket;
        }

        if (event_count)
        {
            event_timer.expires_from_now(yplatform::time_traits::seconds(1));
            event_timer.async_wait(
                std::bind(&module_using_tvm::periodic_event, this, std::placeholders::_1));
        }
    }

    std::shared_ptr<ymod_tvm::tvm2_module> tvm2;
    yplatform::time_traits::timer event_timer{ *yplatform::global_net_reactor->io() };
    int event_count = 3;
};

int main()
{
    std::ofstream of("/tmp/test_tvm_secret");
    of << "esbNM_UIxZIDW5c4ND5zeg";
    of.close();

    yplatform::configuration conf;
    conf.load_from_str(CONFIG);
    yplatform::log::init_global_log(conf);
    yplatform::init_global_reactors(conf);
    yplatform::init_global_repository(conf);

    auto reactor = yplatform::global_net_reactor;
    {
        module_using_tvm module;
        module.init(yplatform::ptree());
        yplatform::repository::instance().start();
        reactor->run();
        std::this_thread::sleep_for(std::chrono::seconds(4));
        yplatform::repository::instance().stop();
        reactor->stop();
        yplatform::repository::instance().fini();
        reactor->fini();
    }

    return 0;
}
