#pragma once
#include <yplatform/active/queue.h>
#include <yplatform/active/task.h>

#include <yplatform/log.h>
#include <yplatform/string_config.h>

#include <boost/asio.hpp>
#include <boost/thread.hpp>
#include <boost/tuple/tuple.hpp>
#include <yplatform/future/future.hpp>
#include <boost/noncopyable.hpp>

namespace yplatform { namespace active {

struct object_context
{
    virtual ~object_context()
    {
    }
};

class method_proxy_base : public element_base
{
public:
    inline method_proxy_base(const prio_t& prio, const ptime& deadline)
        : element_base(prio, deadline)
    {
    }

    virtual ~method_proxy_base()
    {
    }
    virtual void execute(object_context*, const string& name) = 0;
    virtual void cancel(const string& name) = 0;
};

template <typename HOOK, typename CANCEL>
class method_proxy : public method_proxy_base
{
public:
    method_proxy(const HOOK& hook, const CANCEL& cancel, const prio_t& prio, const ptime& deadline)
        : method_proxy_base(prio, deadline)
        , hook_(hook)
        , cancel_(cancel)
        , create_time_(boost::posix_time::microsec_clock::local_time())
    {
    }

protected:
    void execute(object_context* ctx, const string& /*name*/)
    {
        this->hook_(ctx);
    }
    void cancel(const string& /*name*/)
    {
        this->cancel_();
    }

private:
    HOOK hook_;
    CANCEL cancel_;
    ptime create_time_;
};

template <typename HOOK, typename CANCEL>
method_proxy<HOOK, CANCEL>* new_active_proxy(
    const HOOK& hook,
    const CANCEL& cancel,
    const prio_t& prio,
    const ptime& deadline)
{
    return new method_proxy<HOOK, CANCEL>(hook, cancel, prio, deadline);
}

using yplatform::future::future;
using yplatform::future::promise;

#define ACTIVE_QUEUE_TYPE deadline_priority_queue
class pool
    : public ACTIVE_QUEUE_TYPE<pool, method_proxy_base*>
    , public task
    , public yplatform::log::contains_logger
    , private boost::noncopyable
{
    friend class queue_core_access;
    typedef ACTIVE_QUEUE_TYPE<pool, method_proxy_base*> aqueue_t;

protected:
    object_context* create_context(void) const
    {
        if (create_ctx_hook_) return create_ctx_hook_();
        else
            return 0;
    }

public:
    pool(size_t sz, const string& name, const time_duration& enqueue_timeout)
        : aqueue_t(sz), task(), opened_(false), name_(name), enqueue_timeout_(enqueue_timeout)
    {
    }

    pool() : aqueue_t(), task(), opened_(false)
    {
        ;
    }

    virtual ~pool()
    {
        assert(0 == this->size());
        this->close();
    }

    const string& name() const
    {
        return name_;
    }

    void create_ctx_hook(boost::function<object_context*()> create_ctx_hook)
    {
        create_ctx_hook_ = create_ctx_hook;
    }

    template <typename HOOK, typename CANCEL>
    bool enqueue_method(
        const HOOK& hook,
        const CANCEL& cancel,
        const prio_t& prio = 1,
        const ptime& deadline = boost::posix_time::pos_infin)
    {
        if (!opened_) return false;

        ptime enqueue_timeout = boost::posix_time::pos_infin;

        if (enqueue_timeout_ != boost::posix_time::pos_infin)
            enqueue_timeout =
                boost::posix_time::microsec_clock::universal_time() + enqueue_timeout_;
        std::unique_ptr<method_proxy<HOOK, CANCEL>> proxy(
            new_active_proxy(hook, cancel, prio, deadline));
        if (this->push(proxy.get(), enqueue_timeout))
        {
            proxy.release();
            return true;
        }
        return false;
    }

    template <typename HOOK, typename CANCEL>
    bool enqueue_method(
        const HOOK& hook,
        const CANCEL& cancel,
        const prio_t& prio,
        const time_duration& deadline)
    {
        if (deadline != boost::posix_time::pos_infin)
        {
            return enqueue_method(
                hook, cancel, prio, boost::posix_time::microsec_clock::universal_time() + deadline);
        }
        else
        {
            return enqueue_method(hook, cancel, prio);
        }
    }

    bool close(void)
    {
        if (!this->opened_) return false;

        // stop timer thread
        delete this->timer_work_;
        this->timer_work_ = 0;

        this->timer_service_.stop();
        this->timer_thread_->interrupt();
        this->timer_thread_->join(); // use timed join instead XXX
        delete this->timer_thread_;
        this->timer_thread_ = 0;

        delete this->timer_;
        this->timer_ = 0;

        this->deactivate();

        this->opened_ = false;
        return this->task::close();
    }

    bool open(unsigned __nthr)
    {
        this->set_function(boost::bind(&pool::svc, this));

        if (!this->task::open(__nthr)) return false;

        this->timer_ = new boost::asio::deadline_timer(this->timer_service_);

        this->timer_work_ = new boost::asio::io_service::work(this->timer_service_);
        this->timer_thread_ = new boost::thread(boost::bind(&pool::run_timer_service, this));
        this->opened_ = true;
        return true;
    }

    void clear()
    {
        assert(!this->opened_);
        method_proxy_base* b;
        while (this->pop_inactive(b))
        {
            std::unique_ptr<method_proxy_base> raii(b);
#ifdef YPLATFORM_DEBUG
            b->cancel(this->name());
#else
            try
            {
                b->cancel(this->name());
            }
            catch (std::exception& e)
            {
                YLOG_LOCAL(error) << "active::pool::clear exception: " << e.what();
            }
            catch (...)
            {
                YLOG_LOCAL(error) << "active::pool::clear unknown exception";
            }
#endif
        }
    }

    void cancel(method_proxy_base* __e)
    {
        __e->cancel(this->name());
    }

    boost::asio::io_service& timer_service()
    {
        return timer_service_;
    }

private:
    void set_deadline(const boost::system_time& __t)
    {
        if (!this->opened_) return;

        this->timer_->expires_at(__t);
        this->timer_->async_wait(
            boost::bind(&pool::timer_handler, this, boost::asio::placeholders::error));
    }

    void run_timer_service(void)
    {
        this->timer_service_.run();
    }

    void timer_handler(const boost::system::error_code& error)
    {
        if (!error)
        {
            std::ostringstream os;
            os << "DEADLINE TIMER TRIGGERED at '"
               << boost::posix_time::microsec_clock::universal_time() << "', clean the queue\n";
            YLOG_LOCAL(error) << os.str();
            this->clean_deadline_queue();
        }
        else if (error != boost::asio::error::operation_aborted)
        {
            YLOG_LOCAL(error) << "DEADLINE TIMER CANCELLED: " << error.message() << "\n";
        }
    }

    void svc(void)
    {
        std::unique_ptr<object_context> ctx(this->create_context());

        method_proxy_base* b;

        try
        {
            while (this->pop(b))
            {
                std::unique_ptr<method_proxy_base> raii(b);
#ifdef YPLATFORM_DEBUG
                b->execute(ctx.get(), this->name());
#else
                try
                {
                    b->execute(ctx.get(), this->name());
                }
                catch (std::exception& e)
                {
                    YLOG_LOCAL(error) << "active::pool exception: " << e.what();
                }
                catch (...)
                {
                    YLOG_LOCAL(error) << "active::pool unknown exception";
                }
#endif
            }
        }
        catch (const queue_inactive& e)
        {
            return;
        }
    }

    bool opened_;
    string name_;
    boost::asio::io_service timer_service_;
    time_duration enqueue_timeout_;
    boost::asio::io_service::work* timer_work_;
    boost::thread* timer_thread_;
    boost::asio::deadline_timer* timer_;
    boost::function<object_context*()> create_ctx_hook_;
};

}}
