#pragma once

#include <boost/date_time/posix_time/posix_time.hpp>

#include <boost/multi_index_container.hpp>
#include <boost/multi_index/ordered_index.hpp>
#include <boost/multi_index/identity.hpp>
#include <boost/multi_index/tag.hpp>
#include <boost/multi_index/mem_fun.hpp>
#include <boost/multi_index/sequenced_index.hpp>
#include <boost/multi_index/global_fun.hpp>

#include <boost/thread.hpp>
#include <boost/exception/all.hpp>

#include <deque>
#include <queue>
#include <stdexcept>

namespace yplatform { namespace active {

namespace mi = boost::multi_index;

typedef unsigned int prio_t;
using boost::posix_time::ptime;
using boost::posix_time::time_duration;

class queue_inactive
    : public std::runtime_error
    , boost::exception
{
public:
    explicit inline queue_inactive(const std::string& __arg) : std::runtime_error(__arg)
    {
    }
};

class element_base
{
public:
    element_base(const prio_t& __p, const ptime& __d) : prio_(__p), deadline_(__d)
    {
    }
    virtual ~element_base()
    {
    }
    const prio_t& priority(void) const
    {
        return this->prio_;
    }
    const ptime& deadline(void) const
    {
        return this->deadline_;
    }

private:
    prio_t prio_;
    ptime deadline_;
};

template <typename T>
struct active_traits
{
    static const prio_t& priority(const T& t)
    {
        return t.priority();
    }
    static const ptime& deadline(const T& t)
    {
        return t.deadline();
    }
};

template <typename T>
struct active_traits<T*>
{
    static const prio_t& priority(const T* t)
    {
        return t->priority();
    }
    static const ptime& deadline(const T* t)
    {
        return t->deadline();
    }
};

template <typename T>
const prio_t& extract_priority(const T& t)
{
    return active_traits<T>::priority(t);
}

template <typename T>
const ptime& extract_deadline(const T& t)
{
    return active_traits<T>::deadline(t);
}

class queue_core_access
{
public:
    template <class D, class T, class C>
    friend class queue;

    template <class Queue>
    static void push_i(Queue& q, const typename Queue::value_type& t)
    {
        q.push_i(t);
    }

    template <class Queue>
    static typename Queue::value_type pop_i(Queue& q)
    {
        return q.pop_i();
    }

    template <class Queue>
    static void set_deadline(Queue& q, const ptime& d)
    {
        q.set_deadline(d);
    }

    template <class Queue, typename Entry>
    static void cancel(Queue& q, Entry& e)
    {
        q.cancel(e);
    }

private:
    queue_core_access();
};

template <class Derived, class T, class Container = std::deque<T>>
class queue
{
private:
    friend class queue_core_access;
    typedef Derived derived_t;

protected:
    typedef Container queue_t;
    typedef boost::mutex mutex_t;
    //  typedef mutex_t::scoped_lock lock_t;

    class lock_t
    {
    public:
        lock_t(const queue* aq) : lock_(aq->mux()){};
        boost::unique_lock<mutex_t>& lock(void)
        {
            return this->lock_;
        }

    private:
        boost::unique_lock<mutex_t> lock_;
    };

public:
    typedef queue<Derived, T, Container> queue_;
    typedef typename queue_t::value_type value_type;

private:
    Derived& derived(void)
    {
        return *static_cast<Derived*>(this);
    }
    const Derived& derived(void) const
    {
        return *static_cast<const Derived*>(this);
    }

public:
    enum
    {
        DEFAULT_CAPACITY = 1024
    };

    inline queue(size_t __cap = DEFAULT_CAPACITY) : capacity_(__cap), active_(true)
    {
    }

    virtual ~queue()
    {
        //    std::cerr << "active::queue::DTOR: current size: " << this->size () << "\n";
    }

    bool push(
        const T& __t,
        const boost::system_time& __timeout = boost::posix_time::microsec_clock::local_time() +
            boost::posix_time::minutes(10));

    //  template <class IdxTag>
    bool pop(
        T& __t,
        const boost::system_time& __timeout = boost::posix_time::microsec_clock::local_time() +
            boost::posix_time::minutes(10));

    bool pop_inactive(T& __t);

    inline size_t capacity(void) const
    {
        return this->capacity_;
    }
    void set_capacity(size_t capacity)
    {
        this->capacity_ = capacity;
    }
    size_t size(void) const;

    void active(bool __on);
    bool active(void) const;

    inline void activate(void)
    {
        this->active(true);
    }
    inline void deactivate(void)
    {
        this->active(false);
    }

    // hook for deadline active queue family
    void clean_deadline_queue(
        const ptime& /*__now*/ = boost::posix_time::microsec_clock::universal_time()){};

    inline bool active_i(void) const
    {
        return this->active_;
    }
    inline void active_i(bool __on)
    {
        this->active_ = __on;
    }

    void throw_if_inactive(
        const std::string& __msg = std::string("operation on inactive active::queue"));

    size_t size_i(void) const;
    bool empty_i(void) const;

    inline bool full_i(void) const
    {
        return this->size_i() >= this->capacity_;
    }

    void push_i(const T& __t);
    T pop_i(void);

    queue_t& q(void)
    {
        return this->q_;
    }
    mutex_t& mux(void) const
    {
        return this->mux_;
    }

    void notify_readers(void)
    {
        this->r_cond_.notify_one();
    }
    void notify_writers(void)
    {
        this->w_cond_.notify_one();
    }

private:
    size_t capacity_;
    queue_t q_;
    bool active_;
    mutable mutex_t mux_;
    boost::condition_variable r_cond_;
    boost::condition_variable w_cond_;
};

template <class D, class T, class C>
void queue<D, T, C>::throw_if_inactive(const std::string& __msg)
{
    if (!this->active_i()) throw queue_inactive(__msg);
}

template <class D, class T, class C>
bool queue<D, T, C>::active(void) const
{
    lock_t lock(this);
    return this->active_i();
}

template <class D, class T, class C>
void queue<D, T, C>::active(bool __on)
{
    lock_t lock(this);

    if (this->active_i() != __on)
    {
        this->active_i(__on);
        this->r_cond_.notify_all();
        this->w_cond_.notify_all();
    }
}

template <class D, class T, class C>
inline size_t queue<D, T, C>::size_i(void) const
{
    return this->q_.size();
}

template <class D, class T, class C>
inline bool queue<D, T, C>::empty_i(void) const
{
    return this->q_.empty();
}

template <class D, class T, class C>
size_t queue<D, T, C>::size(void) const
{
    lock_t lock(this);
    return this->size_i();
}

template <class D, class T, class C>
T queue<D, T, C>::pop_i(void)
{
    T t = this->q_.front();
    this->q_.pop_front();
    return t;
}

template <class D, class T, class C>
void queue<D, T, C>::push_i(const T& __t)
{
    this->q_.push_back(__t);
}

template <class D, class T, class C>
bool queue<D, T, C>::push(const T& t, const boost::system_time& __timeout)
{
    lock_t lock(this);

    this->throw_if_inactive("push on inactive active::queue");

    while (this->full_i())
    {
        bool rc = this->w_cond_.timed_wait(lock.lock(), __timeout);
        this->throw_if_inactive("push on inactive active::queue");
        if (!rc && __timeout != boost::posix_time::pos_infin) return false;
    }

    queue_core_access::push_i(this->derived(), t);
    this->r_cond_.notify_one();
    return true;
}

template <class D, class T, class C>
// template <class IdxTag>
bool queue<D, T, C>::pop(T& __t, const boost::system_time& __timeout)
{
    lock_t lock(this);
    this->throw_if_inactive("pop on inactive active::queue");

    while (this->empty_i())
    {
        bool rc = this->r_cond_.timed_wait(lock.lock(), __timeout);
        this->throw_if_inactive("pop on inactive active::queue");
        // timeout expired
        if (!rc && __timeout != boost::posix_time::pos_infin) return false;
    }

    __t = queue_core_access::pop_i(this->derived());

    this->w_cond_.notify_one();
    return true;
}

template <class D, class T, class C>
bool queue<D, T, C>::pop_inactive(T& __t)
{
    lock_t lock(this);

    assert(!this->active_i());

    if (this->empty_i())
    {
        return false;
    }

    __t = queue_core_access::pop_i(this->derived());

    return true;
}

struct priority
{
};

template <
    class Derived,
    class T,
    class Container = mi::multi_index_container<
        T,
        mi::indexed_by<mi::ordered_non_unique<
            mi::tag<priority>,
            mi::global_fun<const T&, const prio_t&, extract_priority<T>>>>>>
class priority_queue
    : public queue<
          //                priority_queue <Derived, T, Container>
          Derived,
          T,
          Container>
{
public:
    typedef priority_queue<Derived, T, Container> priority_queue_;
    typedef queue<Derived, T, Container> queue_;

private:
    friend class queue_core_access;

private:
    Derived& derived(void)
    {
        return *static_cast<Derived*>(this);
    }
    const Derived& derived(void) const
    {
        return *static_cast<const Derived*>(this);
    }

public:
    inline priority_queue(size_t __cap = queue_::DEFAULT_CAPACITY) : queue_(__cap)
    {
    }

    void push_i(const T& __t);
    T pop_i(void);
};

template <class D, class T, class C>
void priority_queue<D, T, C>::push_i(const T& __t)
{
    this->q().insert(__t);
}

template <class D, class T, class C>
T priority_queue<D, T, C>::pop_i(void)
{
    typedef typename queue_::queue_t::template index<priority>::type prio_index;
    typename prio_index::const_iterator i = this->q().template get<priority>().begin();

    assert(i != this->q().template get<priority>().end());
    T t = *i;
    this->q().template get<priority>().erase(i);
    return t;
}

struct deadline
{
};

template <
    class Derived,
    class T,
    class Container = mi::multi_index_container<
        T,
        mi::indexed_by<
            mi::sequenced<>,
            mi::ordered_non_unique<
                mi::tag<deadline>,
                mi::global_fun<const T&, const ptime&, extract_deadline<T>>>>>,
    template <class, class, class> class Base = queue>
class deadline_queue
    : public Base<
          //                deadline_queue <Derived, T, Container, Base>
          Derived,
          T,
          Container>
{
public:
    typedef deadline_queue<Derived, T, Container, Base> deadline_queue_;
    typedef Base<Derived, T, Container> queue_;

protected:
    friend class queue_core_access;
    void push_i(const T& __t);
    T pop_i(void);
    inline void cancel(T& /*__t*/
    )
    {
    }

private:
    Derived& derived(void)
    {
        return *static_cast<Derived*>(this);
    }
    const Derived& derived(void) const
    {
        return *static_cast<const Derived*>(this);
    }

public:
    inline deadline_queue(size_t __cap = queue_::DEFAULT_CAPACITY)
        : queue_(__cap), deadline_(boost::posix_time::not_a_date_time)
    {
    }

    void clean_deadline_queue(
        const ptime& __now = boost::posix_time::microsec_clock::universal_time());

    void check_and_set_deadline(void);

    ptime deadline_;
};

#define TemplateB                                                                                  \
    template <class, class, class>                                                                 \
    class B
template <class D, class T, class C, TemplateB>
void deadline_queue<D, T, C, B>::clean_deadline_queue(const ptime& __now)
{
    typedef std::vector<typename queue_::queue_t::value_type> deadlined_methods_t;
    deadlined_methods_t to_cancel;
    {
        typename queue_::lock_t lock(this);
        typedef typename queue_::queue_t::template index<deadline>::type deadline_index;

        typename deadline_index::iterator b = this->q().template get<deadline>().begin();
        typename deadline_index::iterator d = b;

        size_t count = 0;

        while (d != this->q().template get<deadline>().end() && (*d)->deadline() <= __now)
        {
            ++d;
            ++count;
        }

        if (count == 0) return;

        to_cancel.reserve(count);
        to_cancel.assign(b, d);

        this->q().template get<deadline>().erase(b, d);

        this->check_and_set_deadline();
        this->notify_writers();
        // std::cerr << "clean_deadline_queue: " << count << " elements removed\n";
    }

    for (typename deadlined_methods_t::iterator it = to_cancel.begin(); it != to_cancel.end(); ++it)
    {
        try
        {
            queue_core_access::cancel(this->derived(), *it);
        }
        catch (...)
        {
        } // TODO: how to report exceptions here?
        delete *it;
    }
}

template <class D, class T, class C, TemplateB>
void deadline_queue<D, T, C, B>::push_i(const T& __t)
{
    this->queue_::push_i(__t);

    if (this->deadline_ > extract_deadline(__t))
        queue_core_access::set_deadline(this->derived(), this->deadline_ = extract_deadline(__t));
}

template <class D, class T, class C, TemplateB>
void deadline_queue<D, T, C, B>::check_and_set_deadline(void)
{
    if (this->q().empty())
    {
        this->deadline_ = boost::posix_time::max_date_time;
    }
    else
    {
        typedef typename queue_::queue_t::template index<deadline>::type deadline_index;
        typename deadline_index::const_iterator d = this->q().template get<deadline>().begin();
        assert(d != this->q().template get<deadline>().end());

        if (extract_deadline(*d) != this->deadline_)
            queue_core_access::set_deadline(
                this->derived(), this->deadline_ = extract_deadline(*d));
    }
}

template <class D, class T, class C, TemplateB>
T deadline_queue<D, T, C, B>::pop_i(void)
{
    T t = this->queue_::pop_i();
    this->check_and_set_deadline();

    return t;
}
#undef TemplateB

template <typename T>
struct deadline_priority_container
{
    typedef mi::multi_index_container<
        T,
        mi::indexed_by<
            mi::ordered_non_unique<
                mi::tag<priority>,
                mi::global_fun<const T&, const prio_t&, extract_priority<T>>>,
            mi::ordered_non_unique<
                mi::tag<deadline>,
                mi::global_fun<const T&, const ptime&, extract_deadline<T>>>>>
        type;
};

//////////////////////////////////////////////////////
template <
    class Derived,
    class T,
    class Container = mi::multi_index_container<
        T,
        mi::indexed_by<
            mi::ordered_non_unique<
                mi::tag<priority>,
                mi::global_fun<const T&, const prio_t&, extract_priority<T>>>,
            mi::ordered_non_unique<
                mi::tag<deadline>,
                mi::global_fun<const T&, const ptime&, extract_deadline<T>>>>>,
    template <class, class, class> class Base = priority_queue>
class deadline_priority_queue
    : public deadline_queue<
          //                deadline_priority_queue <Derived, T, Container, Base>
          Derived,
          T,
          Container,
          Base>
{
public:
    // self
    typedef deadline_priority_queue<Derived, T, Container, Base> deadline_priority_queue_;

    // base
    typedef deadline_queue<Derived, T, Container, Base> queue_;

protected:
    friend class queue_core_access;
    void push_i(const T& t)
    {
        this->queue_::push_i(t);
    }
    T pop_i(void)
    {
        return this->queue_::pop_i();
    }
    void set_deadline(const ptime& d)
    {
        this->queue_::set_deadline(d);
    }

public:
    inline deadline_priority_queue(size_t __cap = queue_::DEFAULT_CAPACITY) : queue_(__cap)
    {
    }
};

} // namespace active
} // namespave yplatform
