#pragma once

#include <yplatform/ptree.h>

namespace yplatform {

/*
  Congestion controller limits concurrency to prevent congestion.
  Implementation is based on TCP slow start algorithm.
*/
class congestion_controller
{
public:
    bool try_run()
    {
        if (running_ >= limit_)
        {
            return false;
        }
        run_task();
        return true;
    }

    void finish_with_congestion()
    {
        bool reset = !tasks_before_reset_;
        finish_task();
        if (reset)
        {
            reset_limit();
            tasks_before_reset_ = running_;
        }
    }

    void finish_without_congestion()
    {
        if (slow_start_phase())
        {
            finish_task();
            increase_limit();
        }
        else
        {
            bool increase = !tasks_before_increase_;
            finish_task();
            if (increase)
            {
                increase_limit();
                tasks_before_increase_ = running_;
            }
        }
    }

    std::size_t limit() const
    {
        return limit_;
    }

    ptree stats() const
    {
        ptree ret;
        ret.put("running", running_);
        ret.put("limit", limit_);
        ret.put("limit_upper_bound", limit_upper_bound_);
        ret.put("threshold", threshold_);
        return ret;
    }

private:
    void run_task()
    {
        ++running_;
        limit_upper_bound_ = std::max(2 * running_, limit_upper_bound_);
    }

    void finish_task()
    {
        if (!running_)
        {
            throw std::runtime_error("finish error: no running tasks");
        }
        --running_;
        if (tasks_before_increase_)
        {
            --tasks_before_increase_;
        }
        if (tasks_before_reset_)
        {
            --tasks_before_reset_;
        }
    }

    void increase_limit()
    {
        limit_ = std::min(limit_ + 1, limit_upper_bound_);
    }

    void reset_limit()
    {
        threshold_ = limit_ / 2;
        limit_ = min_limit_;
        limit_upper_bound_ = min_limit_;
    }

    bool slow_start_phase()
    {
        return !threshold_ || limit_ < threshold_;
    }

    static const std::size_t min_limit_ = 2;

    std::size_t limit_ = min_limit_;
    std::size_t limit_upper_bound_ = min_limit_;
    std::size_t running_ = 0;
    std::size_t threshold_ = 0;
    std::size_t tasks_before_reset_ = 0;
    std::size_t tasks_before_increase_ = 0;
};

using congestion_controller_ptr = std::shared_ptr<congestion_controller>;

}
