#pragma once

#include <yplatform/time_traits.h>

namespace yplatform {

/*
 * leaky_bucket is a counter whose value "leak" over time.
 * More formally, it subtracts *leak_factor* from value every *leak_interval_ms* milliseconds.
 * Template parameter V should implement operator+=(V), operator-=(V), operator*(int64_t),
 * operator<(V), operator bool().
 */
template <typename V = uint64_t>
class leaky_bucket
{
public:
    leaky_bucket(V leak_factor, uint64_t leak_interval_ms)
        : leak_factor_(leak_factor)
        , leak_interval_ms_(leak_interval_ms)
        , value_()
        , update_ts_(time_traits::clock::now())
    {
    }

    const V get()
    {
        update();
        return value_;
    }

    void add(const V& value)
    {
        update();
        value_ += value;
    }

private:
    void update()
    {
        if (!leak_factor_ || !leak_interval_ms_)
        {
            return;
        }
        auto ms_since_last_update = time_traits::duration_cast<time_traits::milliseconds>(
                                        time_traits::clock::now() - update_ts_)
                                        .count();
        auto intervals_count = ms_since_last_update / leak_interval_ms_;
        V leaked_value = intervals_count * leak_factor_;
        update_ts_ += intervals_count * time_traits::milliseconds(leak_interval_ms_);
        if (leaked_value < value_)
        {
            value_ -= leaked_value;
        }
        else
        {
            value_ = V();
        }
    }

    V leak_factor_;
    uint64_t leak_interval_ms_;
    V value_;
    time_traits::time_point update_ts_;
};

}
