#pragma once

#include <yplatform/algorithm/leaky_bucket.h>
#include <unordered_map>

namespace yplatform {

/*
 * leaky_map is a key-value storage whose values "leak" over time.
 * For more detailed description see leaky_bucket.
 */
template <typename K = std::string, typename V = uint64_t>
class leaky_map
{
public:
    leaky_map(const V& leak_factor, uint64_t leak_interval_ms, uint64_t cleanup_interval_ms = 60000)
        : leak_factor_(leak_factor)
        , leak_interval_ms_(leak_interval_ms)
        , cleanup_interval_(time_traits::milliseconds(cleanup_interval_ms))
        , last_cleanup_ts_(time_traits::clock::now())
    {
    }

    const V get(const K& key)
    {
        erase_zero_elements_if_needed();
        auto it = elements_.find(key);
        if (it == elements_.end())
        {
            return V();
        }
        return it->second.get();
    }

    void add(const K& key, const V& value)
    {
        erase_zero_elements_if_needed();
        auto it = elements_.find(key);
        if (it == elements_.end())
        {
            leaky_bucket bucket(leak_factor_, leak_interval_ms_);
            bucket.add(value);
            elements_.emplace(key, std::move(bucket));
            return;
        }
        it->second.add(value);
    }

    std::size_t size() const
    {
        return elements_.size();
    }

private:
    using leaky_bucket = yplatform::leaky_bucket<V>;

    void erase_zero_elements_if_needed()
    {
        if (time_traits::clock::now() - last_cleanup_ts_ < cleanup_interval_)
        {
            return;
        }
        last_cleanup_ts_ = time_traits::clock::now();
        for (auto it = elements_.begin(); it != elements_.end();)
        {
            if (it->second.get())
            {
                ++it;
            }
            else
            {
                it = elements_.erase(it);
            }
        }
    }

    std::unordered_map<K, leaky_bucket> elements_;
    V leak_factor_;
    uint64_t leak_interval_ms_;
    time_traits::duration cleanup_interval_;
    time_traits::time_point last_cleanup_ts_;
};

}
