#pragma once

#include <boost/multi_index_container.hpp>
#include <boost/multi_index/sequenced_index.hpp>
#include <boost/multi_index/hashed_index.hpp>
#include <boost/multi_index/member.hpp>

namespace yplatform {

namespace mi = boost::multi_index;

// default size calculation is just elements count
template <typename Key, typename Value>
struct lru_entry_size
{
    size_t operator()(const Key& /*k*/, const Value& /*v*/) const
    {
        return 1;
    }
};

template <typename Key, typename Value, typename EntrySize = lru_entry_size<Key, Value>>
class lru_cache
{
public:
    lru_cache(size_t capacity, const EntrySize& entry_size = EntrySize())
        : capacity(capacity), entry_size(entry_size)
    {
    }

    std::optional<Value> get(const Key& key)
    {
        auto it = index_by_key().find(key);
        if (it == index_by_key().end()) return {};
        move_to_front(it);
        return it->val;
    }

    void put(const Key& key, Value val)
    {
        if (entry_size(key, val) > capacity) return;

        auto key_it = index_by_key().find(key);
        if (key_it == index_by_key().end())
        {
            add_entry(key, std::move(val));
        }
        else
        {
            update_entry(key_it, std::move(val));
            move_to_front(key_it);
        }

        if (size > capacity) drop_too_old();
    }

private:
    struct entry
    {
        Key key;
        Value val;
    };

    using container_type = boost::multi_index_container<
        entry,
        mi::indexed_by<mi::sequenced<>, mi::hashed_unique<mi::member<entry, Key, &entry::key>>>>;
    using index_by_pos_type = typename container_type::template nth_index<0>::type;
    using iterator_by_pos = typename index_by_pos_type::iterator;
    using index_by_key_type = typename container_type::template nth_index<1>::type;
    using iterator_by_key = typename index_by_key_type::iterator;

    index_by_key_type& index_by_key()
    {
        return container.template get<1>();
    }

    void move_to_front(iterator_by_key key_it)
    {
        auto pos_it = container.template project<0>(key_it);
        move_to_front(pos_it);
    }

    void move_to_front(iterator_by_pos it)
    {
        container.relocate(container.begin(), it);
    }

    std::pair<iterator_by_pos, bool> add_entry(const Key& key, Value&& val)
    {
        auto new_entry_size = entry_size(key, val);
        auto [it, added] = container.push_front(entry{ key, std::move(val) });
        if (added) size += new_entry_size;
        return std::pair{ it, added };
    }

    bool update_entry(iterator_by_key it, Value&& val)
    {
        return index_by_key().modify(
            it, [val = std::move(val)](entry& e) mutable { e.val = std::move(val); });
    }

    void drop_too_old()
    {
        while (size > capacity)
        {
            drop_entry(std::prev(container.end()));
        }
    }

    void drop_entry(iterator_by_pos it)
    {
        size -= entry_size(it->key, it->val);
        container.erase(it);
    }

    container_type container;
    size_t capacity = 0;
    size_t size = 0;
    EntrySize entry_size;
};

}
