#pragma once

#include <yplatform/application/repository.h>
#include <yplatform/application/context_repository.h>

#include <boost/asio.hpp>

namespace yplatform {

class app_service : public boost::asio::io_service::service
{
public:
    static boost::asio::io_service::id id;

    app_service(boost::asio::io_service& io)
        : boost::asio::io_service::service(io)
        , repository_(std::make_shared<yplatform::repository>())
        , context_repository_(std::make_shared<yplatform::context_repository>())
    {
    }

    yplatform::repository& repository()
    {
        if (!repository_)
        {
            throw std::runtime_error("app_service::repository is null");
        }
        return *repository_;
    }

    void repository(std::shared_ptr<yplatform::repository> repo)
    {
        repository_ = repo;
    }

    yplatform::context_repository& context_repository()
    {
        if (!context_repository_)
        {
            throw std::runtime_error("app_service::context_repository is null");
        }
        return *context_repository_;
    }

    void context_repository(std::shared_ptr<yplatform::context_repository> repo)
    {
        context_repository_ = repo;
    }

private:
    std::shared_ptr<yplatform::repository> repository_;
    std::shared_ptr<yplatform::context_repository> context_repository_;
};

inline void store_module_repository(
    boost::asio::io_service& io,
    std::shared_ptr<yplatform::repository> repo)
{
    auto& service = boost::asio::use_service<app_service>(io);
    service.repository(repo);
}

inline void store_context_repository(
    boost::asio::io_service& io,
    std::shared_ptr<yplatform::context_repository> repo)
{
    auto& service = boost::asio::use_service<app_service>(io);
    service.context_repository(repo);
}

inline yplatform::context_repository& get_context_repository(boost::asio::io_service& io)
{
    auto& service = boost::asio::use_service<app_service>(io);
    return service.context_repository();
}

inline yplatform::context_repository& get_context_repository(reactor& reactor)
{
    return get_context_repository(*reactor.io());
}

inline yplatform::repository& get_module_repository(boost::asio::io_service& io)
{
    auto& service = boost::asio::use_service<app_service>(io);
    return service.repository();
}

inline yplatform::repository& get_module_repository(reactor& reactor)
{
    return get_module_repository(*reactor.io());
}

template <typename T>
inline void register_module(
    boost::asio::io_service& io,
    const std::string& name,
    const std::shared_ptr<T> module)
{
    get_module_repository(io).add_service<T>(name, module);
}

template <typename T>
inline void register_module(reactor& reactor, const std::string& name, std::shared_ptr<T> module)
{
    register_module(*reactor.io(), name, module);
}

} // namespace yplatform
