#pragma once

#include <yplatform/application/config/yaml_to_ptree.h>
#include <boost/algorithm/string.hpp>
#include <boost/property_tree/xml_parser.hpp>
#include <boost/property_tree/info_parser.hpp>
#include <boost/regex.hpp>
#include <fstream>
#include <set>
#include <unordered_set>

// XXX why utils?
namespace utils { namespace config {

class loader
{
public:
    /**
     * Load config to ptree from file 'filename'.
     * Recursively loads base configs, specified in node 'base'.
     * Each config file is considered to have at most one base config.
     * Reads environment name from a file and recursively replaces nodes from
     * service key 'choose_by_env'.
     */
    static void from_file(const string& filename, ptree& config, bool filter = true);

    /**
     * Load YML config from a string.
     */
    static void from_str(const string& source, ptree& config);

private:
    typedef std::pair<ptree::const_assoc_iterator, ptree::const_assoc_iterator> node_range_t;
    static void read_ptree(const string& filename, ptree& config);
    static void filter_by_env(ptree& config);
    static void recursive_filter_by_env(
        ptree& tmp,
        const ptree& cfg,
        const std::string& name,
        const string& environment);
    static std::string find_key_by_env(const ptree& cfg, const std::string& environment);

    /**
     * Overload nodes in base that are specified in derived.
     */
    static ptree merge(const ptree& base, const ptree& derived);
    static void merge_array(
        ptree& res,
        const ptree& base,
        const ptree& derived,
        const std::string& key);
    static void merge_array_with_attrs(
        ptree& res,
        const ptree& base,
        const node_range_t& overloaded,
        const std::string& key);
};

void loader::from_file(const string& filename, ptree& config, bool filter)
{
    std::unordered_set<string> read_files;

    read_ptree(filename, config);
    read_files.insert(filename);

    while (true)
    {
        boost::optional<string> base_filename = config.get_optional<string>("base");
        if (base_filename)
        {
            if (read_files.count(base_filename.get()))
            {
                std::ostringstream error;
                error << "cycle detected in configs inheritance: duplicate read from '"
                      << base_filename.get() << "'";
                throw std::logic_error(error.str());
            }
            config.erase("base");
            ptree base;
            read_ptree(base_filename.get(), base);
            config = merge(base, config);
            read_files.insert(base_filename.get());
        }
        else
        {
            break;
        }
    }

    if (filter) filter_by_env(config);
}

void loader::from_str(const string& source, ptree& config)
{
    yaml_to_ptree::convert_str(source, config);
}

void loader::filter_by_env(ptree& config)
{
    if (auto env_node = config.get_child_optional("environment"))
    {
        string environment = "other";
        std::string env_file_name = env_node->data();
        std::ifstream env_file;
        env_file.open(env_file_name);
        if (env_file.good())
        {
            env_file >> environment;
        }
        if (!env_file.good())
        {
            throw std::runtime_error("failed to read environment file \"" + env_file_name + "\"");
        }
        env_node->data() = environment;
        ptree tmp;
        recursive_filter_by_env(tmp, config.get_child("config"), "config", environment);
        std::swap(tmp, config);
    }
}

void loader::recursive_filter_by_env(
    ptree& tmp,
    const ptree& cfg,
    const std::string& name,
    const string& environment)
{
    if (auto selector = cfg.get_child_optional("choose_by_env"))
    {
        std::string target_key = find_key_by_env(*selector, environment);
        if (target_key.empty())
        {
            // or throw std::domain_error("choose_by_env failed to find environment key for \"" +
            // name + "\"");
            tmp.add_child(name, ptree());
            return;
        }
        auto range = selector->equal_range(target_key);
        for (auto it = range.first; it != range.second; ++it)
        {
            tmp.add_child(name, it->second);
        }
    }
    else
    {
        if (cfg.empty())
        {
            // add only data node
            tmp.add_child(name, cfg);
            return;
        }
        auto& tmp_subtree = tmp.add_child(name, ptree());
        tmp_subtree.put_value(cfg.data());
        for (auto& inner : cfg)
        {
            if (inner.first == "<xmlattr>")
            {
                tmp_subtree.add_child(inner.first, inner.second);
                continue;
            }
            recursive_filter_by_env(tmp_subtree, inner.second, inner.first, environment);
        }
    }
}

std::string loader::find_key_by_env(const ptree& cfg, const std::string& environment)
{
    if (cfg.get_child_optional(environment))
    {
        return environment;
    }
    return cfg.get_child_optional("other") ? "other" : "";
}

void loader::read_ptree(const string& filename, ptree& config)
{
    if (boost::algorithm::ends_with(filename, ".yml"))
    {
        yaml_to_ptree::convert(filename, config);
    }
    else if (boost::algorithm::ends_with(filename, ".info"))
    {
        boost::property_tree::info_parser::read_info(filename, config);
    }
    else
    {
        boost::property_tree::read_xml(filename, config);
    }
}

ptree loader::merge(const ptree& base, const ptree& derived)
{
    ptree result;
    result = base;

    boost::optional<std::string> derived_value = derived.get_value_optional<std::string>();
    if (derived_value)
    {
        result.put_value(derived_value.get());
    }

    std::set<std::string> performed_keys;
    for (ptree::const_iterator it = derived.begin(); it != derived.end(); it++)
    {
        if (performed_keys.count(it->first)) continue;

        // no such node at all
        if (base.count(it->first) == 0)
        {
            node_range_t nodes_range = derived.equal_range(it->first);
            result.insert(result.end(), nodes_range.first, nodes_range.second);
            performed_keys.insert(it->first);
            continue;
        }

        bool is_array = is_node_array(base, it->first) || is_node_array(derived, it->first);
        if (is_array)
        {
            merge_array(result, base, derived, it->first);
            performed_keys.insert(it->first);
        }
        else
        {
            ptree merged_node = merge(base.get_child(it->first), it->second);
            boost::optional<ptree&> subres = result.get_child_optional(it->first);
            if (subres)
            {
                subres.get() = merged_node;
            }
            else
            {
                result.put_child(it->first, merged_node);
            }
            performed_keys.insert(it->first);
        }
    }

    return result;
}

inline bool is_attrs_empty(boost::optional<const ptree&> attrs)
{
    return !attrs || (attrs->size() == 1 && attrs->count("_array"));
}

inline bool is_attrs_empty(const ptree& node)
{
    boost::optional<const ptree&> attrs = node.get_child_optional("<xmlattr>");
    return is_attrs_empty(attrs);
}

void loader::merge_array(
    ptree& res,
    const ptree& base,
    const ptree& derived,
    const std::string& key)
{
    node_range_t overloaded = derived.equal_range(key);

    bool merge_with_attrs = false;
    for (ptree::const_assoc_iterator it = overloaded.first;
         it != overloaded.second && !merge_with_attrs;
         it++)
    {
        if (!is_attrs_empty(it->second)) merge_with_attrs = true;
    }

    if (merge_with_attrs)
    {
        // merge array elements
        merge_array_with_attrs(res, base, overloaded, key);
    }
    else
    {
        // replace array
        res.erase(key);
        res.insert(res.end(), overloaded.first, overloaded.second);
    }
}

inline bool attrs_equal_with_base(const ptree& base, boost::optional<const ptree&> attrs2)
{
    boost::optional<const ptree&> attrs1 = base.get_child_optional("<xmlattr>");

    return (is_attrs_empty(base) && not is_attrs_empty(attrs2)) ||
        (attrs1 && attrs2 && attrs1.get() == attrs2.get());
}

inline bool attrs_equal(const ptree& node, boost::optional<const ptree&> attrs2)
{
    boost::optional<const ptree&> attrs1 = node.get_child_optional("<xmlattr>");
    return attrs1 && attrs2 && attrs1.get() == attrs2.get() && !is_attrs_empty(attrs1);
}

inline boost::optional<const ptree&> find_base_element(
    const ptree& node,
    const std::string& key,
    boost::optional<const ptree&> key_attrs)
{
    for (ptree::const_iterator it = node.begin(); it != node.end(); it++)
    {
        if (it->first != key) continue;
        if (attrs_equal_with_base(it->second, key_attrs))
        {
            return it->second;
        }
    }
    return boost::optional<const ptree&>();
}

inline ptree::iterator find_element_to_replace(
    ptree& node,
    const std::string& key,
    boost::optional<const ptree&> key_attrs)
{
    for (ptree::iterator it = node.begin(); it != node.end(); it++)
    {
        if (it->first != key) continue;
        if (attrs_equal(it->second, key_attrs))
        {
            return it;
        }
    }
    return node.end();
}

void loader::merge_array_with_attrs(
    ptree& res,
    const ptree& base,
    const node_range_t& overloaded,
    const std::string& key)
{
    // remove defaults from result tree
    for (ptree::iterator it = res.begin(); it != res.end();)
    {
        if (it->first == key && is_attrs_empty(it->second))
        {
            it = res.erase(it);
        }
        else
        {
            it++;
        }
    }

    for (ptree::const_assoc_iterator it = overloaded.first; it != overloaded.second; it++)
    {
        boost::optional<const ptree&> attrs = it->second.get_child_optional("<xmlattr>");

        if (is_attrs_empty(attrs)) continue;

        boost::optional<const ptree&> base_node = find_base_element(base, it->first, attrs);

        if (not base_node)
        {
            res.add_child(it->first, it->second);
        }
        else
        {
            ptree::iterator res_node = find_element_to_replace(res, it->first, attrs);
            if (res_node == res.end())
            {
                res.insert(res_node, std::make_pair(it->first, merge(base_node.get(), it->second)));
            }
            else
            {
                res_node->second = merge(base_node.get(), it->second);
            }
        }
    }
}

}}
