#pragma once

#include <yaml-cpp/yaml.h>
#include <boost/property_tree/ptree.hpp>
#include <boost/optional.hpp>
#include <boost/range/iterator_range.hpp>
#include <fstream>
#include <ostream>
#include <string>

#define INNER_TEXT_KEY "__text"
#define SEP "/"

namespace utils { namespace config {

using boost::property_tree::ptree;
using std::string;

class yaml_to_ptree
{
    typedef ptree::path_type path_t;

public:
    static void convert(const std::string& path, ptree& config);
    static void convert_str(const std::string& source, ptree& config);
    static std::ostream& dump(std::ostream& out, const ptree& node);

private:
    static void convert(const YAML::Node& node, ptree& config);
    static path_t get_path(const std::string& name);
    static path_t get_attr_path(const std::string& name);
    static path_t get_xml_node_path(const std::string& name);
    static YAML::Node load_file(const std::string& path);

    static void dump(YAML::Emitter& out, const ptree& node);
};

inline void yaml_to_ptree::convert(const std::string& path, ptree& config)
{
    YAML::Node yaml_config = load_file(path);
    convert(yaml_config, config);
}

inline void yaml_to_ptree::convert_str(const std::string& source, ptree& config)
{
    YAML::Node yaml_config = YAML::Load(source);
    convert(yaml_config, config);
}

inline bool is_node_array(const ptree& node, const std::string& key)
{
    return node.count(key) > 1 ||
        node.get<bool>(ptree::path_type(key + SEP "<xmlattr>" SEP "_array", SEP[0]), false);
}

inline void yaml_to_ptree::convert(const YAML::Node& node, ptree& config)
{
    switch (node.Type())
    {
    case YAML::NodeType::Map:
        for (auto it = node.begin(); it != node.end(); ++it)
        {
            std::string raw_key;
            raw_key = it->first.as<std::string>();
            const YAML::Node current_node = it->second;
            path_t path = get_xml_node_path(raw_key);
            if (current_node.Type() == YAML::NodeType::Sequence)
            {
                for (auto& item : current_node)
                {
                    ptree& item_node = (raw_key == "<<") ? config : config.add_child(path, ptree());
                    item_node.put(get_attr_path("_array"), true);
                    convert(item, item_node);
                }
            }
            else
            {
                convert(
                    current_node,
                    (raw_key == "<<" || raw_key == INNER_TEXT_KEY ?
                         config :
                         config.put_child(path, ptree())));
            }
        }
        break;
    case YAML::NodeType::Sequence:
        for (auto& array_item : node)
        {
            ptree& array_item_node = config.add_child("", ptree());
            convert(array_item, array_item_node);
            array_item_node.put(get_attr_path("_array"), true);
        }
        break;
    case YAML::NodeType::Null:
        config.put_value("");
        break;
    case YAML::NodeType::Scalar:
    {
        std::string value;
        value = node.as<std::string>();
        config.put_value(value);
    }
    break;
    default:
        throw std::runtime_error("unsupported config format");
    }
}

inline yaml_to_ptree::path_t yaml_to_ptree::get_path(const std::string& name)
{
    return { name, SEP[0] };
}

inline yaml_to_ptree::path_t yaml_to_ptree::get_attr_path(const std::string& name)
{
    return { "<xmlattr>" SEP + name, SEP[0] };
}

inline yaml_to_ptree::path_t yaml_to_ptree::get_xml_node_path(const std::string& name)
{
    if (name.substr(0, 1) == "_" && name != INNER_TEXT_KEY) return get_attr_path(name.substr(1));
    return get_path(name);
}

inline YAML::Node yaml_to_ptree::load_file(const std::string& path)
{
    try
    {
        return YAML::LoadFile(path);
    }
    catch (const YAML::Exception& e)
    {
        using namespace std::string_literals;
        throw std::runtime_error("failed to read yaml from file \""s + path + "\": "s + e.what());
    }
}

inline std::ostream& yaml_to_ptree::dump(std::ostream& out, const ptree& node)
{
    YAML::Emitter emitter;
    emitter.SetIndent(4);
    dump(emitter, node);
    return out << emitter.c_str();
}

inline void yaml_to_ptree::dump(YAML::Emitter& out, const ptree& node)
{
    bool is_map = not node.empty();
    boost::optional<string> opt_value = node.get_value_optional<string>();

    if (is_map)
    {
        out << YAML::BeginMap;
        std::set<std::string> performed_keys;
        for (auto& child_pair : node)
        {
            const string& key = child_pair.first;
            const ptree& child = child_pair.second;

            if (performed_keys.count(key)) continue;

            if (key == "<xmlattr>")
            {
                for (auto& attr_node : child)
                {
                    if (attr_node.first == "_array") continue;
                    out << YAML::Key << ("_" + attr_node.first) << YAML::Value;
                    yaml_to_ptree::dump(out, attr_node.second);
                }
                continue;
            }

            out << YAML::Key << key << YAML::Value;

            if (is_node_array(node, key))
            {
                out << YAML::BeginSeq;
                for (auto array_item : boost::make_iterator_range(node.equal_range(key)))
                {
                    yaml_to_ptree::dump(out, array_item.second);
                }
                out << YAML::EndSeq;
            }
            else
            {
                yaml_to_ptree::dump(out, child);
            }

            performed_keys.insert(key);
        }
        if (opt_value and not opt_value.get().empty())
        {
            out << YAML::Key << INNER_TEXT_KEY << YAML::Value << opt_value.get();
        }
        out << YAML::EndMap;
    }
    else
    {
        if (opt_value and not opt_value.get().empty()) out << opt_value.get();
        else
        {
            out << YAML::BeginMap << YAML::EndMap;
        }
    }
}

#undef SEP
#undef INNER_TEXT_KEY

}}
