#pragma once

#include <yplatform/reactor/reactor.h>
#include <yplatform/config.h>
#include <yplatform/ptree.h>
#include <boost/noncopyable.hpp>
#include <boost/function.hpp>
#include <boost/lexical_cast.hpp>
#include <list>
#include <map>
#include <grp.h>
#include <pwd.h>
#include <stdexcept>
#include <sys/types.h>
#include <unistd.h>

namespace yplatform {

class configuration
{
public:
    typedef ::yplatform::ptree ptree;
    typedef std::list<string> lib_path_lst;
    struct module_data
    {
        string name;
        string type;
        string factory;
        ptree options;
        string log_id;
        ptree log_cfg;
    };
    typedef std::list<module_data> module_list;

    configuration();

    void load_from_file(const string& path);
    void load_from_str(const string& source);

    const ptree& log_cfg() const
    {
        return log_cfg_;
    }

    const module_list& modules() const
    {
        return modules_;
    }

    const lib_path_lst& lib_paths() const
    {
        return lib_paths_;
    }

    unsigned verbose() const
    {
        return verbose_;
    }

    uid_t uid() const
    {
        return uid_;
    }

    gid_t gid() const
    {
        return gid_;
    }

    const string& dir() const
    {
        return dir_;
    }

    const string pid() const
    {
        return pid_;
    }

    bool daemon() const
    {
        return daemon_;
    }

    const std::map<std::string, yplatform::reactor_cfg>& io_cfg() const
    {
        return io_cfg_;
    }

    const string& environment() const
    {
        return environment_;
    }

private:
    typedef std::map<string, boost::function<void(configuration&, const string&)>>
        system_config_tab_t;

    void load_configuration(const ptree& cfg);
    void load_module_configuration(const ptree& cfg);
    void load_module_data(const ptree& pt, module_data& data);
    void load_module_system_data(const ptree& pt, module_data& data);
    void load_module_config_data(const ptree& pt, module_data& data);
    void load_module_log_data(const ptree& pt, module_data& data);
    void load_config_data(const ptree& source, ptree& target);
    void load_log_data(const ptree& source, ptree& target);

    void parse_daemon(const string& val)
    {
        daemon_ = boost::lexical_cast<bool>(val);
    }
    void parse_verbose(const string& val)
    {
        verbose_ = boost::lexical_cast<unsigned>(val);
    }
    void parse_uid(const string& val)
    {
        if (val.empty()) return;
        passwd* pwd_info = getpwnam(val.c_str());
        if (pwd_info) uid_ = pwd_info->pw_uid;
        else
            throw std::runtime_error("invalid user name");
    }
    void parse_gid(const string& val)
    {
        if (val.empty()) return;
        group* gr_info = getgrnam(val.c_str());
        if (gr_info) gid_ = gr_info->gr_gid;
        else
            throw std::runtime_error("invalid group name");
    }
    void parse_dir(const string& val)
    {
        dir_ = val;
    }
    void parse_pid(const string& val)
    {
        pid_ = val;
    }
    void parse_libpath(const string& val)
    {
        lib_paths_.push_back(val);
    }

    string include_path(string path);

    ptree log_cfg_;
    module_list modules_;
    system_config_tab_t tab_;

    lib_path_lst lib_paths_;
    bool daemon_;
    unsigned verbose_;
    uid_t uid_;
    gid_t gid_;
    string dir_;
    string pid_;
    std::map<std::string, yplatform::reactor_cfg> io_cfg_;
    string environment_;
};

}
