#pragma once

#include <yplatform/application/detail/module.h>
#include <yplatform/configuration.h>
#include <yplatform/log/find.h>

#include <functional>
#include <map>
#include <memory>
#include <string>
#include <type_traits>

namespace yplatform { namespace detail {

template <typename Base, typename Concrete>
struct factory_impl
{
    std::shared_ptr<module_wrapper<Base>> operator()(
        reactor_set_ptr reactors,
        const configuration::module_data& data)
    {
        static_assert(
            std::is_constructible_v<Concrete, reactor_set&, ptree> ||
                std::is_constructible_v<Concrete, reactor&, ptree> ||
                std::is_constructible_v<Concrete, boost::asio::io_service&, ptree> ||
                std::is_constructible_v<Concrete, reactor_set&> ||
                std::is_constructible_v<Concrete, reactor&> ||
                std::is_constructible_v<Concrete, boost::asio::io_service&> ||
                std::is_constructible_v<Concrete, ptree> || std::is_constructible_v<Concrete>,
            "module must have corresponding constructor");
        test_module<Concrete>();
        std::shared_ptr<Concrete> module;
        auto& conf = data.options.get_child("options");
        if constexpr (std::is_constructible_v<Concrete, reactor_set&, ptree>)
        {
            module = std::make_shared<Concrete>(*reactors, conf);
        }
        else if constexpr (std::is_constructible_v<Concrete, reactor&, ptree>)
        {
            module = std::make_shared<Concrete>(get_reactor(reactors, data.name, conf), conf);
        }
        else if constexpr (std::is_constructible_v<Concrete, boost::asio::io_service&, ptree>)
        {
            module = std::make_shared<Concrete>(get_io(reactors, data.name, conf), conf);
        }
        else if constexpr (std::is_constructible_v<Concrete, reactor_set&>)
        {
            module = std::make_shared<Concrete>(*reactors);
        }
        else if constexpr (std::is_constructible_v<Concrete, reactor&>)
        {
            module = std::make_shared<Concrete>(get_reactor(reactors, data.name, conf));
        }
        else if constexpr (std::is_constructible_v<Concrete, boost::asio::io_service&>)
        {
            module = std::make_shared<Concrete>(get_io(reactors, data.name, conf));
        }
        else if constexpr (std::is_constructible_v<Concrete, ptree>)
        {
            module = std::make_shared<Concrete>(conf);
        }
        else if constexpr (std::is_constructible_v<Concrete>)
        {
            module = std::make_shared<Concrete>();
        }
        else
        {
            throw std::runtime_error(
                "module \"" + data.name + "\" doesn't have corresponding constructor");
        }

        module->name(data.name);
        auto logger = log::find(*reactors->get_global()->io(), data.log_id);
        logger.set_log_prefix(data.name);
        module->logger(logger);

        if constexpr (is_initiable_from_ptree_method<Concrete>::value)
        {
            module->init(conf);
        }
        else if constexpr (is_initiable<Concrete>::value)
        {
            module->init();
        }

        return std::make_shared<module_wrapper_impl<Base, Concrete>>(module);
    }

private:
    reactor& get_reactor(reactor_set_ptr reactors, const std::string& name, const ptree& conf)
    {
        if (auto reactor = conf.get_optional<std::string>("reactor"))
        {
            if (reactors->exists(reactor.get()))
            {
                return *reactors->get(reactor.get());
            }
        }
        if (reactors->exists(name))
        {
            return *reactors->get(name);
        }
        auto reactor = reactors->get_global();
        if (!reactor)
        {
            throw std::runtime_error("reactor not found");
        }
        return *reactor;
    }

    boost::asio::io_service& get_io(
        reactor_set_ptr reactors,
        const std::string& name,
        const ptree& conf)
    {
        return *get_reactor(reactors, name, conf).io();
    }
};

// Factory is constructed with a module base class template parameter and then the concrete module
// type is specified through the set method. It allows you to create containers of polymorphic
// factories
template <typename Base>
class factory
{
public:
    using wrapper_type = detail::module_wrapper<Base>;

    template <typename Concrete>
    void set()
    {
        impl_ =
            std::bind(factory_impl<Base, Concrete>(), std::placeholders::_1, std::placeholders::_2);
    }

    std::shared_ptr<wrapper_type> create(
        reactor_set_ptr reactors,
        const configuration::module_data& data)
    {
        return impl_(reactors, data);
    }

private:
    std::function<std::shared_ptr<wrapper_type>(reactor_set_ptr, const configuration::module_data&)>
        impl_;
};

}}
