#pragma once

#include <yplatform/reactor.h>
#include <yplatform/ptree.h>
#include <memory>
#include <type_traits>

namespace yplatform { namespace detail {

template <typename T>
struct module_wrapper
{
    module_wrapper(std::shared_ptr<T> module) : module(module)
    {
    }
    virtual void start() = 0;
    virtual void stop() = 0;
    virtual void fini() = 0;
    virtual void reload(const ptree&) = 0;
    virtual ~module_wrapper() = default;

    std::shared_ptr<T> module;
};

template <typename T, typename = void>
struct is_initiable : std::false_type
{
};

template <typename T>
struct is_initiable<T, std::void_t<decltype(std::declval<T>().init())>> : std::true_type
{
};

template <typename T, typename = void>
struct is_initiable_from_ptree_method : std::false_type
{
};

template <typename T>
struct is_initiable_from_ptree_method<T, std::void_t<decltype(std::declval<T>().init(ptree()))>>
    : std::true_type
{
};

template <typename T, typename = void>
struct is_startable : std::false_type
{
};

template <typename T>
struct is_startable<T, std::void_t<decltype(std::declval<T>().start())>> : std::true_type
{
};

template <typename T, typename = void>
struct is_stopable : std::false_type
{
};

template <typename T>
struct is_stopable<T, std::void_t<decltype(std::declval<T>().stop())>> : std::true_type
{
};

template <typename T, typename = void>
struct is_finiable : std::false_type
{
};

template <typename T>
struct is_finiable<T, std::void_t<decltype(std::declval<T>().fini())>> : std::true_type
{
};

template <typename T, typename = void>
struct is_reloadable : std::false_type
{
};

template <typename T>
struct is_reloadable<T, std::void_t<decltype(std::declval<T>().reload(ptree()))>> : std::true_type
{
};

struct tests_stub
{
    typedef int init;
    typedef int start;
    typedef int stop;
    typedef int fini;
    typedef int reload;
};

// Checks if the module has the appropriate method (even private). Need to test methods accidentally
// marked private
template <typename U>
struct module_tests
    : U
    , tests_stub
{
    template <typename T = module_tests, typename = typename T::init>
    static std::false_type test_init(int);
    static std::true_type test_init(float);

    template <typename T = module_tests, typename = typename T::start>
    static std::false_type test_start(int);
    static std::true_type test_start(float);

    template <typename T = module_tests, typename = typename T::stop>
    static std::false_type test_stop(int);
    static std::true_type test_stop(float);

    template <typename T = module_tests, typename = typename T::fini>
    static std::false_type test_fini(int);
    static std::true_type test_fini(float);

    template <typename T = module_tests, typename = typename T::reload>
    static std::false_type test_reload(int);
    static std::true_type test_reload(float);
};

template <typename T>
using has_init = std::integral_constant<bool, decltype(module_tests<T>::test_init(0)){}>;

template <typename T>
using has_start = std::integral_constant<bool, decltype(module_tests<T>::test_start(0)){}>;

template <typename T>
using has_stop = std::integral_constant<bool, decltype(module_tests<T>::test_stop(0)){}>;

template <typename T>
using has_fini = std::integral_constant<bool, decltype(module_tests<T>::test_fini(0)){}>;

template <typename T>
using has_reload = std::integral_constant<bool, decltype(module_tests<T>::test_reload(0)){}>;

template <typename Base, typename Concrete>
struct module_wrapper_impl : module_wrapper<Base>
{
    module_wrapper_impl(std::shared_ptr<Base> module) : module_wrapper<Base>(module)
    {
    }

    void start() override
    {
        if constexpr (is_startable<Concrete>::value)
        {
            dynamic_cast<Concrete&>(*module_wrapper<Base>::module).start();
        }
    }

    void stop() override
    {
        if constexpr (is_stopable<Concrete>::value)
        {
            dynamic_cast<Concrete&>(*module_wrapper<Base>::module).stop();
        }
    }

    void fini() override
    {
        if constexpr (is_finiable<Concrete>::value)
        {
            dynamic_cast<Concrete&>(*module_wrapper<Base>::module).fini();
        }
    }

    void reload(const ptree& conf) override
    {
        if constexpr (is_reloadable<Concrete>::value)
        {
            dynamic_cast<Concrete&>(*module_wrapper<Base>::module).reload(conf);
        }
    }
};

template <typename T>
void test_module()
{
    // A reactor can have an io_service that contains a shared_ptr with a repository of modules
    // If the module will store a shared_ptr to the reactor then there will be a circular reference:
    // module -> reactor -> io_service -> repository -> module
    static_assert(
        !std::is_constructible_v<T, std::shared_ptr<reactor_set>, ptree> &&
            !std::is_constructible_v<T, boost::shared_ptr<reactor_set>, ptree> &&
            !std::is_constructible_v<T, std::shared_ptr<reactor>, ptree> &&
            !std::is_constructible_v<T, boost::shared_ptr<reactor>, ptree> &&
            !std::is_constructible_v<T, std::shared_ptr<reactor_set>> &&
            !std::is_constructible_v<T, boost::shared_ptr<reactor_set>> &&
            !std::is_constructible_v<T, std::shared_ptr<reactor>> &&
            !std::is_constructible_v<T, boost::shared_ptr<reactor>>,
        "module must not receive reactors by shared_ptr, use reference instead");

    static_assert(
        (!std::is_constructible_v<T, reactor_set&, ptree> &&
         !std::is_constructible_v<T, reactor&, ptree> &&
         !std::is_constructible_v<T, boost::asio::io_service&, ptree> &&
         !std::is_constructible_v<T, ptree>) ||
            !is_initiable_from_ptree_method<T>::value,
        "module must not have constructor from ptree and method init(ptree) at the same time");

    // Protection against methods accidentally marked private
    static_assert(
        is_initiable<T>::value || is_initiable_from_ptree_method<T>::value || !has_init<T>::value,
        "module must not have init except public init() or init(ptree) methods");

    static_assert(
        is_startable<T>::value || !has_start<T>::value,
        "module must not have start except public start() method");

    static_assert(
        is_stopable<T>::value || !has_stop<T>::value,
        "module must not have stop except public stop() method");

    static_assert(
        is_finiable<T>::value || !has_fini<T>::value,
        "module must not have fini except public fini() method");

    static_assert(
        is_reloadable<T>::value || !has_reload<T>::value,
        "module must not have reload except public reload(ptree) method");
}

} // namespace detail
} // namespace yplatform
