#pragma once

#include <yplatform/application/module.h>
#include <yplatform/application/repository.h>
#include <yplatform/application/context_repository.h>
#include <yplatform/application/app_service.h>
#include <yplatform/reactor/reactor.h>
#include <yplatform/log/logger.h>
#include <yplatform/log/typed.h>
#include <yplatform/log/find.h>
#include <yplatform/util/shared_ptr_cast.h>
#include <memory>
#include <stdexcept>

namespace yplatform {
namespace detail {

template <typename T>
using is_reactor = std::is_same<T, reactor>;

template <typename T>
using is_logger = std::integral_constant<
    bool,
    std::is_same<T, log::source>::value || std::is_same<T, log::tskv_logger>::value>;

template <template <class...> class Pointer>
using is_any_shared_ptr = std::integral_constant<
    bool,
    std::is_same<Pointer<int>, std::shared_ptr<int>>::value ||
        std::is_same<Pointer<int>, boost::shared_ptr<int>>::value>;

template <class T, template <class...> class Pointer>
inline Pointer<T> find_module(repository& repository, const string& name, bool nothrow)
{
    repository::module_ptr ptr = repository.find(name);
    if (!ptr)
    {
        if (nothrow) return Pointer<T>();
        else
            throw std::runtime_error("module \"" + name + "\" not found");
    }

    std::shared_ptr<T> ret = std::dynamic_pointer_cast<T>(ptr);
    if (!ret)
    {
        if (nothrow) return Pointer<T>();
        else
            throw std::runtime_error("module \"" + name + "\" not found");
    }

    return shared_ptr_cast<Pointer>::from(ret);
}

}

template <class T>
inline bool exists_module(const string& name)
{
    repository::module_ptr ptr = repository::instance().find(name);
    if (ptr) return !!std::dynamic_pointer_cast<T>(ptr);
    return false;
}

template <
    class T,
    template <class...> class Pointer = boost::shared_ptr,
    typename std::enable_if<detail::is_any_shared_ptr<Pointer>::value>::type* = nullptr>
inline Pointer<T> find_module(const string& name, bool nothrow = false)
{
    return detail::find_module<T, Pointer>(repository::instance(), name, nothrow);
}

template <
    class T,
    template <class...> class Pointer = boost::shared_ptr,
    typename std::enable_if<detail::is_any_shared_ptr<Pointer>::value>::type* = nullptr>
inline Pointer<T> find_module(boost::asio::io_service& io, const string& name, bool nothrow = false)
{
    return detail::find_module<T, Pointer>(get_module_repository(io), name, nothrow);
}

template <class T>
T find_logger(const string& name);

template <class T>
T find_logger(boost::asio::io_service&, const string& name);

template <>
inline yplatform::log::source find_logger(const string& name)
{
    return yplatform::log::find<yplatform::log::source>(name);
}

template <>
inline yplatform::log::source find_logger(boost::asio::io_service& io, const string& name)
{
    return yplatform::log::find<yplatform::log::source>(io, name);
}

template <>
inline yplatform::log::tskv_logger find_logger(const string& name)
{
    return yplatform::log::find<yplatform::log::tskv_logger>(name);
}

template <>
inline yplatform::log::tskv_logger find_logger(boost::asio::io_service& io, const string& name)
{
    return yplatform::log::find<yplatform::log::tskv_logger>(io, name);
}

template <
    template <class...> class Pointer = boost::shared_ptr,
    typename std::enable_if<detail::is_any_shared_ptr<Pointer>::value>::type* = nullptr>
inline Pointer<reactor> find_reactor(const string& name)
{
    if (!global_reactor_set) throw std::runtime_error("global reactor set is not inited");

    return shared_ptr_cast<Pointer>::from(global_reactor_set->get(name));
}

template <
    class T,
    template <class...> class Pointer = boost::shared_ptr,
    typename std::enable_if<
        !detail::is_reactor<T>::value && !detail::is_logger<T>::value &&
        detail::is_any_shared_ptr<Pointer>::value>::type* = nullptr>
inline Pointer<T> find(const string& name)
{
    return find_module<T, Pointer>(name);
}

template <
    class T,
    template <class...> class Pointer = boost::shared_ptr,
    typename std::enable_if<
        !detail::is_reactor<T>::value && !detail::is_logger<T>::value &&
        detail::is_any_shared_ptr<Pointer>::value>::type* = nullptr>
inline Pointer<T> find(boost::asio::io_service& io, const string& name)
{
    return find_module<T, Pointer>(io, name);
}

template <
    class T,
    template <class...> class Pointer = boost::shared_ptr,
    typename std::enable_if<
        detail::is_reactor<T>::value && detail::is_any_shared_ptr<Pointer>::value>::type* = nullptr>
inline Pointer<reactor> find(const string& name)
{
    return shared_ptr_cast<Pointer>::from(find_reactor(name));
}

template <class T, typename std::enable_if<detail::is_logger<T>::value>::type* = nullptr>
inline T find(const string& name)
{
    return find_logger<T>(name);
}

template <class T, typename std::enable_if<detail::is_logger<T>::value>::type* = nullptr>
inline T find(boost::asio::io_service& io, const string& name)
{
    return find_logger<T>(io, name);
}

template <
    class T,
    typename std::enable_if<!detail::is_reactor<T>::value && !detail::is_logger<T>::value>::type* =
        nullptr>
inline bool exists(const string& name)
{
    return exists_module<T>(name);
}

template <class T, typename std::enable_if<detail::is_reactor<T>::value>::type* = nullptr>
inline bool exists(const string& name)
{
    return global_reactor_set->exists(name);
}

template <class T, typename std::enable_if<detail::is_logger<T>::value>::type* = nullptr>
inline bool exists(const string& /*name*/)
{
    // global_service always returns global logger if requested was not found.
    return log::detail::global_service::instance().inited();
}

}
