#pragma once

#include <yplatform/api.h>
#include <yplatform/ptree.h>
#include <yplatform/log.h>

#include <boost/thread/mutex.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/make_shared.hpp>
#include <boost/enable_shared_from_this.hpp>

namespace yplatform {

class module_stats;
typedef boost::shared_ptr<module_stats> module_stats_ptr;

class API_PUBLIC module
    : public yplatform::log::contains_logger
    , public std::enable_shared_from_this<module>
{
public:
    virtual const module_stats_ptr get_module_stats() const // deprecated
    {
        return module_stats_ptr();
    }

    virtual ptree get_stats() const
    {
        return ptree();
    }

    virtual ~module()
    {
    }

    const std::string& name() const
    {
        return obj_name_;
    }

    void name(const std::string& obj_name)
    {
        obj_name_ = obj_name;
    }

private:
    std::string obj_name_;
};

class module_stats
{
public:
    typedef boost::shared_ptr<ptree> ptree_ptr;
    typedef boost::mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;

    virtual ~module_stats()
    {
    }

    ptree_ptr get_stat() const
    {
        lock_t lock(mux());
        return core_get_stat();
    }

    mutex_t& mux() const
    {
        return mux_;
    }

protected:
    virtual ptree_ptr core_get_stat() const
    {
        return boost::make_shared<ptree>();
    }

private:
    mutable mutex_t mux_;
};

class module_task_context
{
public:
    typedef boost::shared_ptr<ptree> ptree_ptr;
    typedef boost::mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;

    virtual ~module_task_context()
    {
    }

    ptree_ptr get_stat() const
    {
        lock_t lock(mux());
        return core_get_stat();
    }

    mutex_t& mux() const
    {
        return mux_;
    }

protected:
    virtual ptree_ptr core_get_stat() const
    {
        ptree_ptr result(new ptree);
        return result;
    }

private:
    mutable mutex_t mux_;
};

template <typename T>
inline auto shared_from(T* this_ptr)
{
    typedef decltype(this_ptr->shared_from_this()) shared_ptr_type;
    typedef typename shared_ptr_type::element_type element_type;
    if constexpr (std::is_same_v<shared_ptr_type, std::shared_ptr<element_type>>)
    {
        return std::shared_ptr<T>(std::dynamic_pointer_cast<T>(this_ptr->shared_from_this()));
    }
    else
    {
        return boost::shared_ptr<T>(boost::dynamic_pointer_cast<T>(this_ptr->shared_from_this()));
    }
}

template <typename T>
inline auto weak_from(T* this_ptr)
{
    typedef decltype(this_ptr->shared_from_this()) shared_ptr_type;
    typedef typename shared_ptr_type::element_type element_type;
    if constexpr (std::is_same_v<shared_ptr_type, std::shared_ptr<element_type>>)
    {
        return std::weak_ptr<T>(std::dynamic_pointer_cast<T>(this_ptr->shared_from_this()));
    }
    else
    {
        return boost::weak_ptr<T>(boost::dynamic_pointer_cast<T>(this_ptr->shared_from_this()));
    }
}

}
