#pragma once

#include <yplatform/config.h>
#include <yplatform/api.h>
#include <boost/chrono.hpp>
#include <boost/date_time.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/noncopyable.hpp>
#include <boost/thread/mutex.hpp>
#include <boost/thread/locks.hpp>
#include <set>

namespace yplatform {

class profile_proc
{
public:
    virtual ~profile_proc()
    {
    }
    //
    // TODO: use raw micro- or nano- seconds instead of many specific clocks.
    //
    virtual void push(const string& name, const boost::posix_time::time_duration& value) = 0;

    virtual void push(
        const string& name,
        const boost::chrono::high_resolution_clock::duration& value) = 0;

    virtual void push(const string& name, std::size_t value) = 0;
};

typedef boost::shared_ptr<profile_proc> profile_proc_ptr;

class profile_repository : boost::noncopyable
{
    typedef boost::mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;
    typedef std::set<profile_proc_ptr> profile_proc_list;

public:
    API_PUBLIC bool add_proc(profile_proc_ptr proc)
    {
        lock_t lock(mux_);
        procs_.insert(proc);
        return true;
    }

    API_PUBLIC bool rem_proc(profile_proc_ptr proc)
    {
        lock_t lock(mux_);
        procs_.erase(proc);
        return true;
    }

    API_PUBLIC operator bool() const
    {
        lock_t lock(mux_);
        return !procs_.empty();
    }

    API_PUBLIC void operator()(const string& name, const boost::posix_time::time_duration& value)
        const
    {
        lock_t lock(mux_);
        if (procs_.empty()) return;
        for (profile_proc_list::const_iterator i = procs_.begin(), i_end = procs_.end(); i != i_end;
             ++i)
        {
            (*i)->push(name, value);
        }
    }

    API_PUBLIC void operator()(
        const string& name,
        const boost::chrono::high_resolution_clock::duration& value) const
    {
        lock_t lock(mux_);
        if (procs_.empty()) return;
        for (profile_proc_list::const_iterator i = procs_.begin(), i_end = procs_.end(); i != i_end;
             ++i)
        {
            (*i)->push(name, value);
        }
    }

    API_PUBLIC void operator()(const string& name, std::size_t value) const
    {
        lock_t lock(mux_);
        if (procs_.empty()) return;
        for (profile_proc_list::const_iterator i = procs_.begin(), i_end = procs_.end(); i != i_end;
             ++i)
        {
            (*i)->push(name, value);
        }
    }

    API_PUBLIC static profile_repository& instance();

private:
    profile_repository();

private:
    mutable mutex_t mux_;
    profile_proc_list procs_;
};

} // namespace yplatform
