#pragma once

#include <yplatform/application/configuration.h>
#include <yplatform/application/module.h>
#include <yplatform/application/detail/module.h>
#include <yplatform/application/detail/factory.h>
#include <yplatform/api.h>
#include <yplatform/spinlock.h>
#include <yplatform/config.h>

#include <boost/noncopyable.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/thread.hpp>
#include <boost/function.hpp>

#include <iosfwd>
#include <deque>
#include <map>
#include <mutex>

namespace yplatform {

// used by module_registrator at global initialization
class global_factory : boost::noncopyable
{
public:
    typedef std::shared_ptr<detail::module_wrapper<module>> module_wrapper;

    template <typename T>
    API_PUBLIC static void add_factory(const string& name)
    {
        // multiple assign is OK because constraints here will break
        // build for some projects
        instance().factories_[name].set<T>();
    }
    API_PUBLIC static module_wrapper create(
        reactor_set_ptr reactors,
        const configuration::module_data&);

private:
    using impl_type = std::map<std::string, detail::factory<module>>;
    static global_factory& instance();
    global_factory() = default;
    ~global_factory() = default;

private:
    impl_type factories_;
};

class repository : boost::noncopyable
{
public:
    typedef std::shared_ptr<module> module_ptr;
    typedef std::shared_ptr<detail::module_wrapper<module>> module_wrapper;
    typedef boost::mutex mutex_t;
    typedef mutex_t::scoped_lock lock_t;

    enum class module_state
    {
        inited,
        started,
        stopped,
        finished
    };

    struct load_map_item
    {
        string name;
        module_wrapper wrapper;
        module_state state;
    };

    struct runtime_map_item
    {
        runtime_map_item(module_wrapper wrapper) : wrapper(wrapper)
        {
        }

        mutable spinlock lock;
        module_wrapper wrapper;

        module_ptr get_module_safe() const
        {
            std::lock_guard<spinlock> guard(lock);
            return wrapper->module;
        }
    };

    typedef std::map<string, runtime_map_item> map_t;
    typedef std::deque<load_map_item> load_map_t;

    API_PUBLIC void init(reactor_set_ptr reactors, const configuration& cfg);
    API_PUBLIC void start();
    API_PUBLIC void reload(const configuration& cfg);
    API_PUBLIC void stop();
    API_PUBLIC void fini();

    template <typename T>
    API_PUBLIC void add_service(const string& name, const module_ptr& module)
    {
        detail::test_module<T>();
        auto wrapper = std::make_shared<detail::module_wrapper_impl<yplatform::module, T>>(module);
        return this->insert_i(name, wrapper);
    }

    API_PUBLIC module_ptr find(const string& __name);

    API_PUBLIC mutex_t& mux(void)
    {
        return this->mux_;
    }
    API_PUBLIC const map_t& get_services(void)
    {
        return this->services_;
    }

    API_PUBLIC static std::shared_ptr<repository> instance_ptr();
    API_PUBLIC static repository& instance();

private:
    void insert_i(const string& __name, const module_wrapper& __optr);
    module_wrapper find_i(const string& __name);
    void drop_i(const string& __name);

    void load_module(reactor_set_ptr reactors, const configuration::module_data& data);
    bool add_service(reactor_set_ptr reactors, const configuration::module_data& data);

private:
    map_t services_;
    load_map_t load_map_;
    std::list<string> libpath_;
    mutable mutex_t mux_;
    std::string dir_;
};

} // namespace yplatform
