#pragma once

#include <yplatform/config.h>
#include <yplatform/ptree.h>
#include <yplatform/module.h>
#include <yplatform/log.h>
#include <boost/thread/recursive_mutex.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/make_shared.hpp>
#include <map>
#include <memory>

namespace yplatform {

class module_task_context;

class task_context
{
public:
    typedef boost::shared_ptr<ptree> ptree_ptr;
    typedef boost::shared_ptr<module_task_context> module_context_ptr;
    typedef std::map<std::string, module_context_ptr> module_data_map;
    typedef boost::recursive_mutex mutex_t;
    typedef boost::unique_lock<mutex_t> lock_t;
    typedef boost::shared_ptr<task_context> ptr;

    explicit task_context(const std::string& val = "", const std::string& request_id = "");
    explicit task_context(const task_context& other);
    virtual ~task_context();

    ptree_ptr get_stat() const;
    virtual const std::string& get_name() const
    {
        static const std::string DEFAULT_NAME = "abstract";
        return DEFAULT_NAME;
    };

    template <typename ModuleDataT>
    boost::shared_ptr<ModuleDataT> get_module_data(const std::string& name)
    {
        lock_t lock(mux());
        return core_get_module_data<ModuleDataT>(name);
    }

    template <typename ModuleDataT>
    const boost::shared_ptr<ModuleDataT> get_module_data(const std::string& name) const
    {
        lock_t lock(mux());
        return core_get_module_data<ModuleDataT>(name);
    }

    bool contains_module_data(const std::string& name) const;
    void erase_module_data(const std::string& name);

    template <typename ModuleDataT>
    bool add_module_data(const std::string& name, boost::shared_ptr<ModuleDataT> data)
    {
        lock_t lock(mux());
        if (core_contains_module_data(name)) return false;
        return core_add_module_data(name, data);
    }

    template <typename ModuleDataT>
    boost::shared_ptr<ModuleDataT> create_module_data(const std::string& name)
    {
        lock_t lock(mux());
        if (core_contains_module_data(name)) return core_get_module_data<ModuleDataT>(name);

        boost::shared_ptr<ModuleDataT> data(new ModuleDataT);
        core_add_module_data(name, data);
        return data;
    }

    mutex_t& mux() const
    {
        return mux_;
    }

    const std::string& uniq_id() const
    {
        return uniq_id_;
    }

    void append_to_uniq_id(const std::string& suffix)
    {
        constexpr char SEPARATOR = '|';
        uniq_id_.reserve(uniq_id_.length() + 1 + suffix.length());
        uniq_id_ += SEPARATOR;
        uniq_id_ += suffix;
    }

    const std::string& request_id() const
    {
        return request_id_;
    }

    void set_request_id(const std::string& request_id)
    {
        request_id_ = request_id;
    }

    log::source& logger() const
    {
        return logger_;
    }

    void begin()
    {
        begin_time_ = boost::posix_time::microsec_clock::local_time();
    }

    const boost::posix_time::ptime& begin_time() const
    {
        return begin_time_;
    }

    const time_traits::time_point& deadline() const
    {
        return deadline_;
    }

    void cancel()
    {
        cancelled_ = true;
    }

    void deadline(const time_traits::time_point& deadline)
    {
        deadline_ = std::min(deadline_, deadline);
    }

    void deadline_from_now(const time_traits::duration& duration)
    {
        deadline_ = std::min(deadline_, time_traits::clock::now() + duration);
    }

    bool is_cancelled() const
    {
        return cancelled_ || time_traits::clock::now() >= deadline_;
    }

    static ptr fake()
    {
        static const auto fake = boost::make_shared<task_context>("unknown-ctx");
        return fake;
    }

protected:
    void init(const std::string& ini = "");
    virtual ptree_ptr core_get_stat() const;

    template <typename ModuleDataT>
    boost::shared_ptr<ModuleDataT> core_get_module_data(const std::string& name)
    {
        module_data_map::const_iterator i = modules_data_.find(name);
        if (i == modules_data_.end()) return boost::shared_ptr<ModuleDataT>();
        return boost::dynamic_pointer_cast<ModuleDataT>(i->second);
    }

    template <typename ModuleDataT>
    const boost::shared_ptr<ModuleDataT> core_get_module_data(const std::string& name) const
    {
        module_data_map::const_iterator i = modules_data_.find(name);
        if (i == modules_data_.end()) return boost::shared_ptr<ModuleDataT>();
        return boost::dynamic_pointer_cast<ModuleDataT>(i->second);
    }

    bool core_contains_module_data(const std::string& name) const
    {
        module_data_map::const_iterator i = modules_data_.find(name);
        return (i != modules_data_.end());
    }

    template <typename ModuleDataT>
    bool core_add_module_data(const std::string& name, boost::shared_ptr<ModuleDataT> data)
    {
        module_context_ptr data_ = boost::static_pointer_cast<module_task_context>(data);
        return modules_data_.insert(std::make_pair(name, data_)).second;
    }

    module_data_map modules_data_;

private:
    mutable mutex_t mux_;

    boost::posix_time::ptime begin_time_;
    bool cancelled_;
    time_traits::time_point deadline_ = time_traits::time_point::max();

    std::string uniq_id_;
    std::string request_id_;

    // Depricated
    mutable yplatform::log::source logger_;
};

typedef boost::shared_ptr<task_context> task_context_ptr;
typedef boost::shared_ptr<const task_context> const_task_context_ptr;

typedef boost::weak_ptr<task_context> task_context_weak_ptr;
typedef boost::weak_ptr<const task_context> const_task_context_weak_ptr;

typedef std::set<task_context_ptr> task_context_list;

}
