#include <yplatform/convert/yconv.h>

#include <boost/scoped_array.hpp>

//#include <iostream>
#include <cstring> // std::memcpy
#include <cerrno>
#include <stdexcept>

namespace yplatform { namespace convert {
namespace detail {
  

template <typename Iterator, typename T>
inline Iterator
copy_buf (Iterator __out, const T* __from, const T* __to)
{
  for (const T* t = __from; t != __to; ++t, ++__out)
    *__out = *t;

  return __out;
}

template <typename Iterator, typename T>
inline Iterator
copy_buf (Iterator __out, Iterator __end, const T* __from, const T* __to)
{
  for (const T* t = __from; t != __to && __out != __end; ++t, ++__out)
    *__out = *t;

  return __out;
}

}

template <class IconvImpl, typename InputIterator, typename OutputIterator>
OutputIterator
iconvert (IconvImpl __impl, size_t __icap, size_t __ocap,
          InputIterator __first, InputIterator __last,
          OutputIterator __result)
{
  typedef char CHAR;

  if (__icap<= 0 || __ocap <= 0) 
    throw bad_args_error ("Invalid args")
      << errno_info (EINVAL)
      << BOOST_ERROR_INFO;

  boost::scoped_array <CHAR> obuf (new CHAR [__ocap]);
  boost::scoped_array <CHAR> ibuf (new CHAR [__icap]);

  size_t isize = 0;
  size_t osize = 0;

  InputIterator iter = __first; 

  for (;;)
  {

    while (iter != __last && isize < __icap)
    {
      ibuf[isize] = *iter;
      ++iter;
      ++isize;
    }

    if (! isize) break;

    CHAR* iptr = ibuf.get ();
    CHAR* optr = &obuf[osize];

    size_t isz = isize;
    size_t osz = __ocap - osize;

    bool rc = __impl (iptr, isz, optr, osz);
    if (! rc) 
      switch (errno) 
      {
        case EILSEQ: 
          throw illegal_sequence_error ("Invalid input sequence")
            << errno_info (errno)
            << BOOST_ERROR_INFO;

        case EINVAL:
          // if no progress in convertation
          if (isz == isize)
          {
            if (isz == __icap) 
              // input buffer is full, and we still have incomplete seq: this should not happen
              throw algo_error ("Too small input buffer")
                << errno_info (errno)
                << BOOST_ERROR_INFO;
            else
              throw illegal_sequence_error ("Truncated input sequence")
                << errno_info (errno)
                << BOOST_ERROR_INFO;
          }

          // otherwise it is OK
      }

    std::memcpy (ibuf.get (), iptr, isz);
    isize = isz;

    __result = detail::copy_buf (__result, obuf.get (), optr);
    osize = 0;
  }


  CHAR* iptr = NULL;
  CHAR* optr = &obuf[osize];
  isize = 0;
  osize = __ocap;

  bool rc = __impl (iptr, isize, optr, osize);
  if (! rc) 
    switch (errno) 
    {
      case EILSEQ:
          throw illegal_sequence_error ("Invalid input sequence")
            << errno_info (errno)
            << BOOST_ERROR_INFO;
    }

  std::string s (&obuf[0], optr);
  return detail::copy_buf (__result, obuf.get (), optr);
}

template <class IconvImpl, typename InputIterator, typename OutputIterator>
OutputIterator
iconvert (IconvImpl __impl, size_t __icap, size_t __ocap,
          InputIterator __first, InputIterator __last,
          OutputIterator __result, OutputIterator __end)
{
  typedef char CHAR;

  if (__icap<= 0 || __ocap <= 0) 
    throw bad_args_error ("Invalid args")
      << errno_info (EINVAL)
      << BOOST_ERROR_INFO;

  boost::scoped_array <CHAR> obuf (new CHAR [__ocap]);
  boost::scoped_array <CHAR> ibuf (new CHAR [__icap]);

  size_t isize = 0;
  size_t osize = 0;

  InputIterator iter = __first; 

  for (;;)
  {

    while (iter != __last && isize < __icap)
    {
      ibuf[isize] = *iter;
      ++iter;
      ++isize;
    }

    if (! isize) break;

    CHAR* iptr = ibuf.get ();
    CHAR* optr = &obuf[osize];

    size_t isz = isize;
    size_t osz = __ocap - osize;

    bool rc = __impl (iptr, isz, optr, osz);
    if (! rc) 
      switch (errno) 
      {
        case EILSEQ:
          throw illegal_sequence_error ("Invalid input sequence")
            << errno_info (errno)
            << BOOST_ERROR_INFO;

        case EINVAL:
          // if no progress in convertation
          if (isz == isize)
          {
            if (isz == __icap) 
              // input buffer is full, and we still have incomplete seq: this should not happen
              throw algo_error ("Too small input buffer")
                << errno_info (errno)
                << BOOST_ERROR_INFO;
            else
              throw illegal_sequence_error ("Truncated input sequence")
                << errno_info (errno)
                << BOOST_ERROR_INFO;
          }

          // otherwise it is OK
      }



    std::memcpy (ibuf.get (), iptr, isz);
    isize = isz;

    __result = detail::copy_buf (__result, __end, obuf.get (), optr);
    osize = 0;
  }

  CHAR* iptr = NULL;
  CHAR* optr = &obuf[osize];
  isize = 0;
  osize = __ocap;

  bool rc = __impl (iptr, isize, optr, osize);
  if (! rc) 
    switch (errno) 
    {
      case EILSEQ:
          throw illegal_sequence_error ("Invalid input sequence")
            << errno_info (errno)
            << BOOST_ERROR_INFO;
    }

  return detail::copy_buf (__result, __end, obuf.get (), optr);
}

}}
