#pragma once

#include <yplatform/detail/coroutine.h>
#include <yplatform/util/tuple_unpack.h>

#ifndef _LIBCPP_HAS_NO_COROUTINES
#include <boost/asio/experimental/co_spawn.hpp>
#include <boost/asio/experimental/detached.hpp>
#include <experimental/coroutine>
#endif

#include <boost/asio/coroutine.hpp>
#include <boost/asio/dispatch.hpp>
#include <boost/asio/post.hpp>
#include <boost/asio/executor.hpp>
#include <boost/asio/io_service.hpp>
#include <memory>
#include <tuple>

namespace yplatform {

namespace detail {
template <typename T>
struct is_awaitable : std::false_type
{
};

template <typename T>
struct is_awaitable_token : std::false_type
{
};
}

#ifndef _LIBCPP_HAS_NO_COROUTINES
using boost::asio::experimental::awaitable;
using boost::asio::experimental::co_spawn;
using boost::asio::experimental::await_token;
#if BOOST_VERSION < 107000
#define use_awaitable co_await boost::asio::experimental::this_coro::token()
#endif // BOOST_VERSION

namespace detail {
template <typename T>
struct is_awaitable_token<await_token<T>> : std::true_type
{
};
template <typename T>
struct is_awaitable<awaitable<T>> : std::true_type
{
};
}
template <typename T>
using async = awaitable<T>;
using async_void = async<void>;
#endif // _LIBCPP_HAS_NO_COROUTINES

template <typename Handler>
constexpr bool is_awaitable()
{
    return detail::is_awaitable<std::remove_reference_t<Handler>>::value;
}

template <typename Handler>
constexpr bool is_awaitable_token()
{
    return detail::is_awaitable_token<std::remove_reference_t<Handler>>::value;
}

template <typename Coroutine>
class yield_context : public boost::asio::coroutine
{
public:
    using exception_type = std::exception_ptr;

    yield_context(std::shared_ptr<Coroutine> ptr, boost::asio::executor executor = {})
        : ptr(ptr), executor(std::move(executor))
    {
    }

    yield_context(const yield_context&) = default;
    yield_context& operator=(const yield_context&) = default;
    yield_context(yield_context&&) = default;
    yield_context& operator=(yield_context&&) = default;
    ~yield_context() = default;

    yield_context respawn()
    {
        return yield_context(ptr, executor);
    }

    template <typename... Args>
    void operator()(Args&&... args) const
    {
        if (executor)
        {
            boost::asio::post(
                executor,
                [ptr = ptr, args = std::make_tuple(*this, std::forward<Args>(args)...)]() mutable {
                    safe_call_with_tuple_args(*ptr, std::move(args));
                });
        }
        else
        {
            safe_call_with_args(*ptr, *this, std::forward<Args>(args)...);
        }
    }

    template <typename T>
    void operator()(future::future<T> future) const
    {
        auto exception = future.get_exception();
        if (exception)
        {
            (*this)(exception);
            return;
        }
        (*this)();
    }

    void operator()(exception_type exception) const
    {
        if (executor)
        {
            boost::asio::post(executor, [ptr = ptr, exception]() mutable {
                call_with_exception_or_throw(*ptr, exception);
            });
        }
        else
        {
            call_with_exception_or_throw(*ptr, exception);
        }
    }

    template <typename... Args>
    auto capture(Args&... args)
    {
        return detail::yield_context_capture(*this, &args...);
    }

private:
    template <typename F, typename... Args>
    static void safe_call_with_args(F& f, Args&&... args)
    {
        try
        {
            f(std::forward<Args>(args)...);
        }
        catch (...)
        {
            call_with_exception_or_throw(f, std::current_exception());
        }
    }

    template <typename F, typename... Args>
    static void safe_call_with_tuple_args(F& f, std::tuple<Args...>&& args)
    {
        try
        {
            util::call_with_tuple_args(f, std::move(args));
        }
        catch (...)
        {
            call_with_exception_or_throw(f, std::current_exception());
        }
    }

    template <typename F>
    static void call_with_exception_or_throw(F& f, std::exception_ptr exception)
    {
        if constexpr (std::is_invocable_v<F, exception_type>)
        {
            f(exception);
        }
        else
        {
            throw;
        }
    }

    std::shared_ptr<Coroutine> ptr;
    boost::asio::executor executor;
};

// Allows to spawn coroutines in two ways:
// 1. simple braced-init-lists syntax
//   spawn(T{param_a, param_b});
// 2. shared_ptr directly
//   auto coro = make_shared<T>();
//   spawn(coro);
// Coroutine will be run on executor.
// Note that io_service is simply threated as executor too.
template <typename Executor, typename T, typename... Args>
inline auto spawn(Executor&& executor, T&& t, Args&&... args)
    -> std::enable_if_t<detail::is_executor_v<std::remove_reference_t<Executor>>>
{
    auto ptr = detail::forward_as_shared(std::forward<T>(t));
    auto ctx = yield_context(ptr, detail::adapt_executor(executor));
    ctx(std::forward<Args>(args)...);
}

// Simply spawn coroutine in current thread.
template <typename T, typename... Args>
inline auto spawn(T&& t, Args&&... args)
    -> std::enable_if_t<!detail::is_executor_v<std::remove_reference_t<T>>>
{
    auto ptr = detail::forward_as_shared(std::forward<T>(t));
    auto ctx = yield_context(ptr);
    ctx(std::forward<Args>(args)...);
}

// Respawn another coroutine.
template <typename T, typename... Args>
inline void spawn(const yield_context<T>& other_ctx, Args&&... args)
{
    auto ctx = yield_context<T>(other_ctx);
    ctx(std::forward<Args>(args)...);
}

}
