#pragma once

#include <yplatform/future/future.hpp>
#include <yplatform/util/tuple_unpack.h>
#include <boost/asio/executor.hpp>
#include <boost/asio/io_service.hpp>
#include <type_traits>

namespace yplatform::detail {

template <typename T>
struct is_shared_ptr : std::false_type
{
};

template <typename T>
struct is_shared_ptr<std::shared_ptr<T>> : std::true_type
{
};

template <class T>
constexpr bool is_shared_ptr_v = is_shared_ptr<T>::value;

template <typename T>
inline auto forward_as_shared(T&& t)
{
    if constexpr (is_shared_ptr_v<std::remove_const_t<std::decay_t<T>>>)
    {
        return t;
    }
    else
    {
        return std::make_shared<T>(std::forward<T>(t));
    }
}

template <class T>
constexpr bool is_executor_v = std::bool_constant<
    boost::asio::is_executor<T>::value || std::is_same_v<T, boost::asio::io_service>>::value;

template <typename T>
inline auto adapt_executor(T&& t)
{
    if constexpr (std::is_same_v<std::remove_reference_t<T>, boost::asio::io_service>)
    {
        return t.get_executor();
    }
    else
    {
        return boost::asio::executor(std::forward<T>(t));
    }
}

template <typename YieldContext, typename... Args>
struct yield_context_capture
{
    YieldContext context;
    std::tuple<Args*...> captured;

    yield_context_capture(YieldContext context, Args*... args) : context(context), captured(args...)
    {
    }

    template <typename... Values>
    void operator()(Values&&... values)
    {
        util::call_with_tuple_args(
            [&](auto... a) { std::tie(*a...) = std::make_tuple(std::forward<Values>(values)...); },
            captured);
        context();
    }

    template <typename T>
    void operator()(future::future<T> future)
    {
        try
        {
            *std::get<0>(captured) = future.get();
        }
        catch (...)
        {
            context(std::current_exception());
            return;
        }
        context();
    }
};

}