#pragma once

#include <yplatform/encoding/iterators/base64_alphabet.h>
#include <boost/config.hpp>
#include <boost/iterator/iterator_adaptor.hpp>
#include <boost/iterator/iterator_traits.hpp>
#include <boost/iterator/transform_iterator.hpp>
#include <cassert>
#include <cstddef> // size_t

#if defined(BOOST_NO_STDC_NAMESPACE)
namespace std {
using ::size_t;
}
#endif

namespace yplatform { namespace iterators {
namespace detail {

template <base64_alphabet Alphabet>
struct base64_from_binary_table;

template <>
struct base64_from_binary_table<base64_alphabet::urlsafe>
{
    static const char* get()
    {
        static const char* AlphabetSafe =
            "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789-_";
        return AlphabetSafe;
    }
};

template <>
struct base64_from_binary_table<base64_alphabet::regular>
{
    static const char* get()
    {
        static const char* AlphabetRegular =
            "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789+/";
        return AlphabetRegular;
    }
};

}
/////////////////////////////////////////////////////////
// convert binary integers to base64 chars
//
template <
    typename Base,
    typename CharType = BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type,
    base64_alphabet Alphabet = base64_alphabet::regular>
class base64_from_binary
    : public boost::iterator_adaptor<
          base64_from_binary<Base, CharType, Alphabet>,
          Base,
          CharType,
          boost::single_pass_traversal_tag,
          CharType>
{
    friend class boost::iterator_core_access;
    typedef base64_from_binary<Base, CharType, Alphabet> this_t;
    typedef boost::
        iterator_adaptor<this_t, Base, CharType, boost::single_pass_traversal_tag, CharType>
            super_t;
    typedef BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type base_value_type;

private:
    CharType dereference() const
    {
        assert(current_index_ < 4);
        return buffer_[current_index_];
    }

    void increment()
    {
        if (++current_index_ == 4) read_next();
    }

    bool equal(this_t const& other) const
    {
        return this->base_reference() == other.base_reference() &&
            this->current_index_ == other.current_index_;
    }

    using table = detail::base64_from_binary_table<Alphabet>;

    void read_next()
    {
        unsigned char in[3] = { 0, 0, 0 };
        int len = -1;
        for (; len < 2 && this->base_reference() != last_; ++this->base_reference())
        {
            in[++len] = *this->base_reference();
        }
        if (len == -1) return;
        current_index_ = Alphabet == base64_alphabet::regular ? 0 : (2 - len);
        buffer_[current_index_] = table::get()[(in[0] & 0xFC) >> 2];
        buffer_[current_index_ + 1] = table::get()[((in[0] & 3) << 4) + ((in[1] & 0xF0) >> 4)];

        if (len > 0)
            buffer_[current_index_ + 2] =
                table::get()[((in[1] & 0xf) << 2) + ((in[2] & 0xC0) >> 6)];
        else if (Alphabet == base64_alphabet::regular)
            buffer_[2] = '=';

        if (len > 1) buffer_[3] = table::get()[in[2] & 0x3f];
        else if (Alphabet == base64_alphabet::regular)
            buffer_[3] = '=';
    }

    char buffer_[4];
    int current_index_;
    Base last_;

public:
    template <typename OtherBase>
    base64_from_binary(
        OtherBase const& start,
        OtherBase const& last // = OtherBase()
        ,
        typename boost::enable_if<boost::is_convertible<OtherBase, Base>, void*>::type = 0)
        : super_t(static_cast<Base>(start)), current_index_(4), last_(last)
    {
        read_next();
    }
};

template <
    typename Base,
    typename CharType = BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type>
using base64_urlsafe_from_binary = base64_from_binary<Base, CharType, base64_alphabet::urlsafe>;

}}
