#pragma once

#include <yplatform/encoding/iterators/base64_alphabet.h>
#include <boost/config.hpp>
#include <boost/iterator/iterator_adaptor.hpp>
#include <boost/iterator/iterator_traits.hpp>
#include <boost/iterator/transform_iterator.hpp>
#include <stdexcept>

namespace yplatform { namespace iterators {
/////////////////////////////////////////////////////////
// convert base64 chars to binary integers
//
namespace detail {

template <base64_alphabet>
struct binary_from_base64_table
{
    static const char* get();
};

template <>
struct binary_from_base64_table<base64_alphabet::regular>
{
    static const char* get()
    {
        // 127 = '=' - pad marker
        static const char lookup_table[] = {
            -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
            -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, 62,
            -1, -1, -1, 63, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1, -1, 0,
            1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22,
            23, 24, 25, -1, -1, -1, -1, -1, -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
            39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1
        };
// metrowerks trips this assertion - how come?
#if !defined(__MWERKS__)
        BOOST_STATIC_ASSERT(128 == sizeof(lookup_table));
#endif
        return lookup_table;
    }
};

template <>
struct binary_from_base64_table<base64_alphabet::urlsafe>
{
    static const char* get()
    {
        // 127 = '=' - pad marker
        static const char lookup_table[] = {
            -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
            -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1, -1,
            -1, 62, -1, -1, 52, 53, 54, 55, 56, 57, 58, 59, 60, 61, -1, -1, -1, -1, -1, -1, -1, 0,
            1,  2,  3,  4,  5,  6,  7,  8,  9,  10, 11, 12, 13, 14, 15, 16, 17, 18, 19, 20, 21, 22,
            23, 24, 25, -1, -1, -1, -1, 63, -1, 26, 27, 28, 29, 30, 31, 32, 33, 34, 35, 36, 37, 38,
            39, 40, 41, 42, 43, 44, 45, 46, 47, 48, 49, 50, 51, -1, -1, -1, -1, -1
        };
// metrowerks trips this assertion - how come?
#if !defined(__MWERKS__)
        BOOST_STATIC_ASSERT(128 == sizeof(lookup_table));
#endif
        return lookup_table;
    }
};

template <typename CharType, base64_alphabet Alphabet>
struct to_6_bit
{
    typedef CharType result_type;

    using table = binary_from_base64_table<Alphabet>;

    result_type operator()(CharType t) const
    {

        signed char value = -1;
        auto as_uint = static_cast<unsigned int>(t);
        if (as_uint <= 127)
        {
            value = table::get()[as_uint];
        }

        if (-1 == value)
        {
            throw std::runtime_error("bad base64 char " + std::to_string(as_uint));
        }
        return value;
    }
};
} // namespace detail

template <
    typename Base,
    typename CharType = BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type,
    base64_alphabet Alphabet = base64_alphabet::regular>
class binary_from_base64
    : public boost::iterator_adaptor<
          binary_from_base64<Base, CharType, Alphabet>,
          Base,
          CharType,
          boost::single_pass_traversal_tag,
          CharType>
{
    friend class boost::iterator_core_access;
    typedef binary_from_base64<Base, CharType, Alphabet> this_t;
    typedef boost::
        iterator_adaptor<this_t, Base, CharType, boost::single_pass_traversal_tag, CharType>
            super_t;
    typedef BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type base_value_type;

private:
    CharType dereference() const
    {
        if (!buffered_) const_cast<this_t*>(this)->read_next();

        return static_cast<CharType>((buf_ >> (8 * (2 - step_))) & 0xff);
    }

    void increment()
    {
        if (!buffered_) read_next();

        if (++step_ > 2)
        {
            step_ = 0;
            buffered_ = false;
            ;
        }
    }

    bool equal(this_t const& other) const
    {
        return this->base_reference() == other.base_reference() && this->step_ == other.step_;
    }

    void read_next()
    {
        buf_ = 0;
        unsigned int istep = 0;
        int stepVal = 0;

        if (stop_) throw std::runtime_error("unexpected input after '=' trailing");

        // read 4 bytes from base, convert into 6 bit and fill buff
        for (buf_ = 0; this->base_reference() != last_ && istep < 4;
             ++(this->base_reference()), ++istep)
        {
            //      std::cerr << "istep=" << istep << " to_6_bit (" << *this->base_reference () <<
            //      ")\n";
            if (*this->base_reference() == '=')
            {
                if (istep < 2) throw std::runtime_error("bad placed '=' in input stream");
                buf_ <<= 6;
                stepVal++;
                stop_ = true;
            }
            else
            {
                if (stop_)
                    throw std::runtime_error("premature end of stream, trailing '=' expected");

                buf_ <<= 6;
                buf_ |= to_6_bit_(*this->base_reference());
            }
        }

        if (istep < 4)
        {
            if (check_trails_)
                throw std::runtime_error("premature end of stream, trailing '=' expected");

            while (istep++ < 4)
            {
                buf_ <<= 6;
                stepVal++;
            }
        }

        step_ = stepVal;
        buf_ >>= (step_ * 8);
        buffered_ = true;
    }

    Base last_;
    uint32_t buf_;
    int step_;
    bool buffered_;
    bool stop_;
    bool check_trails_;
    const detail::to_6_bit<CharType, Alphabet> to_6_bit_;

public:
    template <typename OtherBase>
    binary_from_base64(
        OtherBase const& first,
        OtherBase const& last // = OtherBase()
        ,
        typename boost::enable_if<boost::is_convertible<OtherBase, Base>, void*>::type = 0)
        : super_t(static_cast<Base>(first))
        , last_(static_cast<Base>(last))
        , step_(0)
        , buffered_(false)
        , stop_(false)
        , check_trails_(false)
        , to_6_bit_(detail::to_6_bit<CharType, Alphabet>())
    {
    }
};

template <
    typename Base,
    typename CharType = BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type>
using binary_from_base64_urlsafe = binary_from_base64<Base, CharType, base64_alphabet::urlsafe>;

}}
