#pragma once

#include <boost/config.hpp>
#include <boost/iterator/iterator_adaptor.hpp>
#include <boost/iterator/iterator_traits.hpp>
#include <boost/iterator/transform_iterator.hpp>
#include <boost/type_traits/remove_const.hpp>
#include <cassert>
#include <stdexcept>
#include <cstddef> // size_t

namespace yplatform { namespace iterators {

//////////////////////////////////////////////////////////////
// translate url_encoded stream to binary
template <
    typename Base
    // output char type
    ,
    typename CharType = BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type>
class binary_from_urlenc
    : public boost::iterator_adaptor<
          binary_from_urlenc<Base, CharType>,
          Base,
          CharType,
          boost::single_pass_traversal_tag,
          CharType>
{
    friend class boost::iterator_core_access;
    typedef boost::iterator_adaptor<
        binary_from_urlenc<Base, CharType>,
        Base,
        CharType,
        boost::single_pass_traversal_tag,
        CharType>
        super_t;

    typedef binary_from_urlenc<Base, CharType> this_t;
    typedef BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type base_value_type;
    typedef typename boost::remove_const<CharType>::type buffer_byte_t;

    CharType hex_value(unsigned char val) const
    {
        if (val <= '9' && val >= '0') return static_cast<CharType>(val - '0');
        else if (val <= 'f' && val >= 'a')
            return static_cast<CharType>(val - 'a' + 10);
        else if (val <= 'F' && val >= 'A')
            return static_cast<CharType>(val - 'A' + 10);
        return static_cast<CharType>(0);
    }

    CharType char_from_hex(unsigned char val1, unsigned char val2) const
    {
        return static_cast<CharType>(16 * hex_value(val1) + hex_value(val2));
    }

    CharType dereference_impl()
    {
        if (!m_converted) convert();
        return m_cvtbuf;
    }

    CharType dereference() const
    {
        return const_cast<this_t*>(this)->dereference_impl();
    }

    bool equal(this_t const& rhs) const
    {
        return this->base_reference() == rhs.base_reference();
    }

    void increment()
    {
        if (this->base_reference() == last_) return;
        ++(this->base_reference());
        m_converted = false;
    }

    void convert()
    {
        m_cvtbuf = *this->base_reference();
        m_converted = true;
        if (m_cvtbuf == '+')
        {
            m_cvtbuf = ' ';
            return;
        }
        if (m_cvtbuf != '%') return;
        if (++(this->base_reference()) == last_) return;
        m_cvtbuf = *this->base_reference();
        if (++(this->base_reference()) == last_) return;
        m_cvtbuf = char_from_hex(m_cvtbuf, *this->base_reference());
    }

    buffer_byte_t m_cvtbuf;
    bool m_converted;
    Base last_;

public:
    template <typename OtherBase>
    binary_from_urlenc(
        OtherBase const& start,
        OtherBase const& last,
        typename boost::enable_if<boost::is_convertible<OtherBase*, Base*>, void*>::type = 0)
        : super_t(static_cast<Base>(start)), m_converted(false), last_(static_cast<Base>(last))
    {
    }
};

}}
