#pragma once

#include <boost/config.hpp>
#include <boost/iterator/iterator_adaptor.hpp>
#include <boost/utility/enable_if.hpp>
#include <boost/type_traits/is_convertible.hpp>

namespace yplatform { namespace iterators {

namespace detail {

template <typename C>
struct crlf_traits
{
    inline static bool is_cr(C const& c)
    {
        return (c == static_cast<C>('\r'));
    }
    inline static bool is_lf(C const& c)
    {
        return (c == static_cast<C>('\n'));
    }

    static C CR;
};

template <typename C>
C crlf_traits<C>::CR __attribute__((weak)) = static_cast<C>('\r');

}

//////////////////////////////////////////////////////////////
// translate single \n to \r\n
template <
    typename Base
    // output char type
    ,
    typename CharType = typename boost::iterator_value<Base>::type,
    typename TraitsType = detail::crlf_traits<CharType>>
class crlf
    : public boost::iterator_adaptor<
          crlf<Base, CharType, TraitsType>,
          Base,
          CharType,
          boost::single_pass_traversal_tag,
          typename boost::iterator_reference<Base>::type
          // , CharType
          >
{
    struct enabler
    {
    };
    typedef TraitsType traits_type;

public:
    inline crlf(traits_type traits = traits_type())
        : crlf::iterator_adaptor_(), traits_(traits), seen_cr_(false), end_(false)
    {
    }

    explicit inline crlf(Base iterator, bool end = false, traits_type traits = traits_type())
        : crlf::iterator_adaptor_(iterator), traits_(traits), seen_cr_(false), end_(end)
    {
    }

    template <typename OtherT>
    crlf(
        crlf<OtherT, CharType, TraitsType> const& other,
        typename boost::enable_if<boost::is_convertible<OtherT, Base>, enabler>::type = enabler())
        : crlf::iterator_adaptor_(other.base)
        , traits_(other.traits_)
        , seen_cr_(other.seen_cr_)
        , end_(other.end_)
    {
    }

private:
    friend class boost::iterator_core_access;

    typename crlf::reference dereference() const
    {
        typename crlf::reference val = *this->base_reference();

        if (!seen_cr_ && traits_.is_lf(val)) return traits_type::CR;

        return val;
    }

    void increment()
    {
        const typename crlf::reference val = *this->base_reference();

        if (!seen_cr_ && traits_.is_lf(val))
        {
            seen_cr_ = true;
            return;
        }

        if (traits_.is_cr(val))
        {
            seen_cr_ = true;
        }
        else
        {
            seen_cr_ = false;
        }

        ++this->base_reference();
    }

    template <typename OtherT>
    bool equal(
        crlf<OtherT, CharType, TraitsType> const& other,
        typename boost::enable_if<boost::is_convertible<OtherT, Base>, enabler>::type =
            enabler()) const
    {
        return (end_ || other.end_ || seen_cr_ == other.seen_cr_) &&
            this->base_reference() == other.base_reference();
    }

    traits_type traits_;
    bool seen_cr_;
    bool end_;
};

}}
