#pragma once

#include <boost/config.hpp>
#include <boost/iterator/iterator_adaptor.hpp>
#include <boost/iterator/iterator_traits.hpp>

namespace yplatform { namespace iterators {

//////////////////////////////////////////////////////////////
// translate from one bits-width to another
template <
    typename Base,
    int BitsOut,
    int BitsIn
    // output char type
    ,
    typename CharType = BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type>
class transform_width
    : public boost::iterator_adaptor<
          transform_width<Base, BitsOut, BitsIn, CharType>,
          Base,
          CharType,
          boost::single_pass_traversal_tag,
          CharType>
{
    friend class boost::iterator_core_access;
    typedef BOOST_DEDUCED_TYPENAME boost::iterator_adaptor<
        transform_width<Base, BitsOut, BitsIn, CharType>,
        Base,
        CharType,
        boost::single_pass_traversal_tag,
        CharType>
        super_t;

    typedef transform_width<Base, BitsOut, BitsIn, CharType> this_t;
    typedef BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type base_value_type;

    CharType fill();

    CharType dereference_impl()
    {
        if (!m_full)
        {
            m_current_value = fill();
            m_full = true;
        }
        return m_current_value;
    }

    CharType dereference() const
    {
        return const_cast<this_t*>(this)->dereference_impl();
    }

    bool equal(this_t const& rhs) const
    {
        return this->base_reference() == rhs.base_reference();
    }

    void increment()
    {
        m_displacement += BitsOut;

        while (m_displacement >= BitsIn)
        {
            m_displacement -= BitsIn;
            if (0 == m_displacement) m_bufferfull = false;

            if (!m_bufferfull)
            {
                if (++(this->base_reference()) == m_end) break;
            }
        }
        m_full = false;
    }

    Base m_end; // end marker
    CharType m_current_value;

    unsigned int m_displacement; // number of bits left in current input buffer
    base_value_type m_buffer;
    bool m_full;       // flag to current output char is ready
    bool m_bufferfull; // flag to indicate that m_buffer has data

public:
    transform_width(Base const& start, Base const& finish = Base())
        : super_t(start), m_end(finish), m_displacement(0), m_full(false), m_bufferfull(false)
    {
    }

    template <typename OtherBase>
    transform_width(
        OtherBase const& start,
        OtherBase const& finish = OtherBase(),
        typename boost::enable_if<boost::is_convertible<OtherBase*, Base*>, void*>::type = 0)
        : super_t(static_cast<Base>(start))
        , m_end(static_cast<Base>(finish))
        , m_displacement(0)
        , m_full(false)
        , m_bufferfull(false)
    {
    }
};

template <typename Base, int BitsOut, int BitsIn, typename CharType>
CharType transform_width<Base, BitsOut, BitsIn, CharType>::fill()
{
    CharType retval = 0;
    unsigned int missing_bits = BitsOut;

    for (;;)
    {
        unsigned int bcount;
        if (!m_bufferfull)
        {
            m_buffer = *this->base_reference();
            m_bufferfull = true;
            bcount = BitsIn;
        }
        else
        {
            bcount = BitsIn - m_displacement;
        }

        unsigned int i = (std::min)(bcount, missing_bits);

        // shift interesting bits to least significant position
        unsigned int j = m_buffer >> (bcount - i);

        // strip off uninteresting bits
        // (note presumption of two's complement arithmetic)
        j &= ~(-(1 << i));

        // append then interesing bits to the output value
        retval <<= i;
        retval |= j;

        missing_bits -= i;

        if (0 == missing_bits) break;

        if (++(this->base_reference()) == m_end) break;
        m_bufferfull = false;
    }
    return retval;
}

}}
