#pragma once

#include <boost/config.hpp>
#include <boost/iterator/iterator_adaptor.hpp>
#include <boost/iterator/iterator_traits.hpp>
#include <boost/iterator/transform_iterator.hpp>
#include <cassert>
#include <stdexcept>
#include <cstddef> // size_t

namespace yplatform { namespace iterators {

//////////////////////////////////////////////////////////////
// translate binary stream to url_encoded
template <
    typename Base
    // output char type
    ,
    typename CharType = BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type>
class urlenc_from_binary
    : public boost::iterator_adaptor<
          urlenc_from_binary<Base, CharType>,
          Base,
          CharType,
          boost::single_pass_traversal_tag,
          CharType>
{
    friend class boost::iterator_core_access;
    typedef boost::iterator_adaptor<
        urlenc_from_binary<Base, CharType>,
        Base,
        CharType,
        boost::single_pass_traversal_tag,
        CharType>
        super_t;

    typedef urlenc_from_binary<Base, CharType> this_t;
    typedef BOOST_DEDUCED_TYPENAME boost::iterator_value<Base>::type base_value_type;

    char hex_digit(unsigned char val) const
    {
        return static_cast<char>(((val < 10) ? '0' : 'a') + (val % 10));
    }

    void convert_input()
    {
        CharType c = *this->base_reference();
        if ((c >= '0' && c <= '9') || (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') ||
            c == '-' || c == '_' || c == '.' || c == '~')
        {
            // conversion is unnecessary
            m_index = -1;
            return;
        }
        if (c == ' ')
        {
            m_index = -2;
            return;
        }
        m_cvtbuf[0] = hex_digit(static_cast<unsigned char>(c) / 16);
        m_cvtbuf[1] = hex_digit(static_cast<unsigned char>(c) % 16);
        m_index = 0;
    }

    CharType dereference_impl()
    {
        if (!m_converted)
        {
            convert_input();
            m_converted = true;
        }

        switch (m_index)
        {
        case -1:
            return *this->base_reference();
        case -2:
            return '+';
        case 0:
            return static_cast<CharType>('%');
        default:
            return m_cvtbuf[m_index - 1];
        }
    }

    CharType dereference() const
    {
        return const_cast<this_t*>(this)->dereference_impl();
    }

    bool equal(this_t const& rhs) const
    {
        return this->base_reference() == rhs.base_reference();
    }

    void increment()
    {
        if (!m_converted) convert_input();

        switch (m_index)
        {
        case -2:
        case -1:
        case 2:
            m_converted = false;
            ++(this->base_reference());
            break;

        default:
            ++m_index;
            break;
        }
    }

    bool m_converted;
    int m_index;
    CharType m_cvtbuf[2];

public:
#if 0
  urlenc_from_binary (Base const& start, Base const& finish = Base ())
    : super_t (start)
    , m_end (finish)
    , m_displacement (0)
    , m_full (false)
    , m_bufferfull (false)
  {}
#endif

    template <typename OtherBase>
    urlenc_from_binary(
        OtherBase const& start
#if 1
        ,
        typename boost::enable_if<boost::is_convertible<OtherBase*, Base*>, void*>::type = 0
#endif
        )
        : super_t(static_cast<Base>(start)), m_converted(false), m_index(-1)
    {
    }
};

}}
