#pragma once

#include <yplatform/detail/error.h>
#include <yplatform/config.h>

namespace yplatform {

template <typename T>
struct is_error_enum : std::false_type
{
};

class error
{
public:
    error() = default;

    error(int val, const boost::system::error_category& cat) : code_(val, cat)
    {
    }

    error(int val, const boost::system::error_category& cat, string message)
        : code_(val, cat), message_(std::make_shared<string>(std::move(message)))
    {
    }

    template <class ErrorEnum>
    error(ErrorEnum e, detail::enable_if_error_enum<ErrorEnum, void>* = nullptr)
        : code_(static_cast<int>(e), error_category(e))
    {
    }

    template <class ErrorEnum>
    error(ErrorEnum e, string message, detail::enable_if_error_enum<ErrorEnum, void>* = nullptr)
        : code_(static_cast<int>(e), error_category(e))
        , message_(std::make_shared<string>(std::move(message)))
    {
    }

    template <class ErrorCode>
    error(ErrorCode&& e, detail::enable_if_error_code<ErrorCode, void>* = nullptr)
        : code_(e.value(), e.category())
    {
        if constexpr (detail::has_method_what<ErrorCode>::value)
        {
            // For compatibility with legacy error codes
            message_ = std::make_shared<string>(e.what());
        }
    }

    template <class ErrorCode>
    error(ErrorCode&& e, string message, detail::enable_if_error_code<ErrorCode, void>* = nullptr)
        : code_(e.value(), e.category()), message_(std::make_shared<string>(std::move(message)))
    {
    }

    error(const error&) = default;

    error& operator=(const error&) = default;

    string message() const
    {
        using namespace std::string_literals;
        string result = code().message();
        if (message_ && message_->size())
        {
            result += ": "s + *message_;
        }
        return result;
    }

    explicit operator bool() const noexcept
    {
        return static_cast<bool>(code());
    }

    bool operator!() const noexcept
    {
        return !static_cast<bool>(code());
    }

    const boost::system::error_code& code() const noexcept
    {
        return code_;
    }

private:
    boost::system::error_code code_;
    std::shared_ptr<string> message_;
};

inline bool operator==(const error& lhs, const error& rhs) noexcept
{
    return lhs.code() == rhs.code();
}

inline bool operator==(const error& lhs, const boost::system::error_condition& rhs) noexcept
{
    return lhs.code() == rhs;
}

inline bool operator==(const error& lhs, const boost::system::error_code& rhs) noexcept
{
    return lhs.code() == rhs;
}

inline bool operator==(const boost::system::error_condition& lhs, const error& rhs) noexcept
{
    return lhs == rhs.code();
}

inline bool operator==(const boost::system::error_code& lhs, const error& rhs) noexcept
{
    return lhs == rhs.code();
}

inline bool operator!=(const error& lhs, const error& rhs) noexcept
{
    return !(lhs == rhs);
}

inline bool operator!=(const error& lhs, const boost::system::error_condition& rhs) noexcept
{
    return !(lhs == rhs);
}

inline bool operator!=(const error& lhs, const boost::system::error_code& rhs) noexcept
{
    return !(lhs == rhs);
}

inline bool operator!=(const boost::system::error_condition& lhs, const error& rhs) noexcept
{
    return !(lhs == rhs);
}

inline bool operator!=(const boost::system::error_code& lhs, const error& rhs) noexcept
{
    return !(lhs == rhs);
}

inline bool operator<(const error& lhs, const error& rhs) noexcept
{
    return lhs.code() < rhs.code();
}

inline std::size_t hash_value(const error& ec)
{
    return hash_value(ec.code());
}

}
