#pragma once
#include <boost/exception/get_error_info.hpp>
#include <boost/exception/info.hpp>
#include <boost/exception/info_tuple.hpp>
#include <boost/tuple/tuple.hpp>
#include <boost/lexical_cast.hpp>
#include <boost/system/error_code.hpp>

#include <exception>
#include <string>
#include <system_error>

namespace yplatform {

struct tag_throw_what
{
};
struct tag_throw_private
{
};
struct tag_throw_public
{
};

typedef ::boost::error_info<struct tag_throw_class, std::string> error_class_info;
typedef ::boost::error_info<struct tag_throw_public, std::string> error_public_info;
typedef ::boost::error_info<struct tag_throw_private, std::string> error_private_info;
typedef ::boost::tuple<error_class_info, error_public_info, error_private_info> error_info;

struct tag_throw_error_code
{
};

typedef ::boost::error_info<struct tag_throw_error_code, int> error_code_info;

typedef ::boost::error_info<struct tag_throw_system, std::error_code> std_system_error;

inline std_system_error system_error(std::error_code const& ec)
{
    return std_system_error(ec);
}

inline std_system_error system_error(std::error_code&& ec)
{
    return std_system_error(std::move(ec));
}

typedef ::boost::error_info<struct tag_throw_system, boost::system::error_code> boost_system_error;

inline boost_system_error system_error(boost::system::error_code const& ec)
{
    return boost_system_error(ec);
}

inline boost_system_error system_error(boost::system::error_code&& ec)
{
    return boost_system_error(std::move(ec));
}

////////////////////////////////////////////////////////////////////////////////
/// Generic exception class.
/** Exception is derived class from boost::exception and std::exception.
 * boost::exception is streaming extended attributes (such as information about
 * functions, files, lines and private messages).
 * std::exception is used as standard c++ exception interface.
 */
class exception
    : public ::boost::exception
    , public ::std::exception
{
public:
    /// Constructs an exception instance.
    /**
     * This constructor creates an exception instance with class
     * <tt>yplatform::exception</tt>.
     */
    inline exception(void)
    {
        *this << error_info(
            std::string("yplatform::exception"), std::string("exception"), std::string());
    }

    /// Constructs an exception instance.
    /**
     * This constructor creates an exception instance with given class.
     * @param cl The exception class.
     */
    explicit inline exception(const std::string& cl)
    {
        *this << error_info(cl, std::string("exception"), std::string());
    }

    /// Constructs an exception instance.
    /**
     * This constructor creates an exception instance with given class and
     * \em public message.
     * @param cl The exception class.
     * @param pub The public exception message.
     */
    inline exception(std::string const& cl, std::string const& pub)
    {
        *this << error_info(cl, pub, std::string());
    }

    /// Constructs an exception instance.
    /**
     * This constructor creates an exception instance with given class and
     * \em public and \em private messages.
     * @param cl The exception class.
     * @param pub The public error message.
     * @param prv The private error message.
     */
    inline exception(std::string const& cl, std::string const& pub, std::string const& prv)
    {
        *this << error_info(cl, pub, prv);
    }

    /// Destructor.
    /**
     * Destructs exception instance.
     */
    virtual inline ~exception()
    {
    }

    /// Get string identifying exception.
    /**
     * Returns a null terminated character sequence that may be used to identify
     * the exception.
     *
     * @returns A pointer to a c-string with content related to the exception.
     *          This is guaranteed to be valid at least until the exception object
     *          from which it is obtained is destroyed or until a non-const member
     *          function of the exception object is called.
     */
    virtual const char* what() const noexcept
    {
        try
        {
            throw_this();
        }
        catch (boost::exception const& e)
        {
            if (std::string const* msg = boost::get_error_info<error_public_info>(e))
                return msg->c_str();
        }
        catch (...)
        {
        }
        return "";
    }

    std::string error_class() const noexcept
    {
        try
        {
            throw_this();
        }
        catch (boost::exception const& e)
        {
            try
            {
                if (std::string const* msg = boost::get_error_info<error_class_info>(e))
                    return *msg;
            }
            catch (...)
            {
            }
        }
        catch (...)
        {
        }
        return std::string();
    }

    std::string error_public() const noexcept
    {
        try
        {
            throw_this();
        }
        catch (boost::exception const& e)
        {
            try
            {
                if (std::string const* msg = boost::get_error_info<error_public_info>(e))
                    return *msg;
            }
            catch (...)
            {
            }
        }
        catch (...)
        {
        }
        return std::string();
    }

    std::string error_private() const noexcept
    {
        try
        {
            throw_this();
        }
        catch (boost::exception const& e)
        {
            try
            {
                if (std::string const* msg = boost::get_error_info<error_private_info>(e))
                    return *msg;
            }
            catch (...)
            {
            }
        }
        catch (...)
        {
        }
        return std::string();
    }

    virtual std::string public_message() const
    {
        return public_message_i(); //  + message_suffix ();
    }

    virtual std::string private_message() const
    {
        return private_message_i() + message_suffix();
    }

    virtual std::exception_ptr exception_ptr() const
    {
        return std::make_exception_ptr(*this);
    }

protected:
    std::string message_suffix() const
    {
        std::string msg;

        try
        {
            throw_this();
        }
        catch (boost::exception const& e)
        {
            // system error
            if (std::error_code const* se = boost::get_error_info<std_system_error>(e))
            {
                msg += ": ";
                msg += se->message();
            }
            // boost system error
            if (boost::system::error_code const* se = boost::get_error_info<boost_system_error>(e))
            {
                msg += ": ";
#if 0 // do not want include asio headers here...
        if (*se == ::boost::asio::error::operation_aborted)
          msg += "Timeout";
        else
#endif
                msg += se->message();
            }

            char const* const* file = boost::get_error_info<::boost::throw_file>(e);
            int const* line = boost::get_error_info<::boost::throw_line>(e);
            // char const* const* func =
            //    boost::get_error_info < ::boost::throw_function> (e);

            if (file && *file)
            {
                msg += ", at ";
                msg += *file;

                if (line)
                {
                    msg += '@';
                    msg += ::boost::lexical_cast<std::string>(*line);
                }

                // XXX function names are too long to be returned or maybe not?
            }
        }
        catch (...)
        {
        }

        return msg;
    }

    std::string public_message_i() const
    {
        return error_public();
    }

    std::string private_message_i() const
    {
        std::string msg(public_message_i());
        std::string prv(error_private());

        if (!prv.empty())
        {
            msg += ": ";
            msg += prv;
        }

        return msg;
    }

private:
    void throw_this() const
    {
        // surprise!
        throw *this;
    }
};

#define YPLATFORM_ERROR_DEF(name, base, cl, xpub)                                                  \
public                                                                                             \
    base                                                                                           \
    {                                                                                              \
        explicit inline name() : base(cl, xpub, std::string())                                     \
        {                                                                                          \
        }                                                                                          \
        inline name(std::string const& pub) : base(cl, pub, std::string())                         \
        {                                                                                          \
        }                                                                                          \
        inline name(std::string const& pub, std::string const& prv) : base(cl, pub, prv)           \
        {                                                                                          \
        }                                                                                          \
        inline name(std::string const& c, std::string const& pub, std::string const& prv)          \
            : base(c, pub, prv)                                                                    \
        {                                                                                          \
        }                                                                                          \
        virtual inline ~name()                                                                     \
        {                                                                                          \
        }                                                                                          \
        virtual std::exception_ptr exception_ptr() const                                           \
        {                                                                                          \
            return std::make_exception_ptr(*this);                                                 \
        }                                                                                          \
    }

struct deadline_error
    : YPLATFORM_ERROR_DEF(deadline_error, exception, "yplatform::exception", "no free workers");

struct std_error : YPLATFORM_ERROR_DEF(std_error, exception, "yplatform::exception", "std error");

#ifndef BOOST_ERROR_INFO
#define BOOST_ERROR_INFO                                                                           \
    ::boost::throw_function(BOOST_CURRENT_FUNCTION)                                                \
        << ::boost::throw_file(__FILE__) << ::boost::throw_line(__LINE__)
#endif

}
