#pragma once

#include <yplatform/future/future_exceptions.hpp>
#include <yplatform/future/future_detail.hpp>
#include <yplatform/time_traits.h>

//  Copyright (c) 2007 Braddock Gaskill Distributed under the Boost
//  Software License, Version 1.0. (See accompanying file
//  LICENSE_1_0.txt or copy at http://www.boost.org/LICENSE_1_0.txt)
//  exception_ptr.hpp/cpp copyright Peter Dimov

// #include <boost/utility/result_of.hpp>
// #include <boost/exception/all.hpp>
#include <boost/type_traits/is_base_of.hpp>
#include <boost/tti/has_member_function.hpp>
#include <boost/intrusive_ptr.hpp>

namespace yplatform { namespace future {

namespace detail {
#pragma GCC diagnostic push
#pragma GCC diagnostic ignored "-Wold-style-cast"
BOOST_TTI_HAS_MEMBER_FUNCTION(exception_ptr)
#pragma GCC diagnostic pop
} // namespace detail

template <typename R>
class promise
{
public:
    using future_type = detail::future_impl;
    using value_type = std::decay_t<R>;
    using container_type = detail::promise_impl<value_type>;

    promise() : impl_(std::make_shared<container_type>()), impl_ref_(impl_.get())
    {
    }

    promise(const promise&) = default;
    promise& operator=(const promise& other)
    {
        promise(other).swap(*this);
        return *this;
    }

    void set(value_type r)
    {
        impl_->f_.set_value(std::move(r), impl_->value_);
    }

    void set_or_throw(value_type r)
    {
        if (!impl_->f_.set_value(std::move(r), impl_->value_)) throw future_already_set();
    }

    // stores the exception e and transitions to ready()
    template <typename E>
    typename boost::enable_if_c<
        detail::has_member_function_exception_ptr<std::exception_ptr (E::*)() const>::value,
        void>::type
    set_exception_core(E const& e)
    {
        impl_->f_.set_exception(e.exception_ptr());
    }

    template <typename E>
    typename boost::disable_if_c<
        detail::has_member_function_exception_ptr<std::exception_ptr (E::*)() const>::value,
        void>::type
    set_exception_core(E const& e)
    {
        impl_->f_.set_exception(std::make_exception_ptr(e));
    }

    template <typename E>
    typename boost::
        disable_if<boost::is_same<typename std::decay<E>::type, std::exception_ptr>, void>::type
        set_exception(E&& e)
    {
        this->set_exception_core(std::forward<E>(e));
    }

    // Attempt's a 'throw', assuming there is an exception set
    void set_current_exception()
    {
        impl_->f_.set_exception(std::current_exception());
    }

    void set_exception(const std::exception_ptr& e)
    {
        impl_->f_.set_exception(e);
    }

    void set_exception(std::exception_ptr&& e)
    {
        impl_->f_.set_exception(std::move(e));
    }

    bool is_needed()
    {
        return impl_->f_.is_needed();
    }
    void wait_until_needed() const
    {
        return impl_->f_.wait_until_needed();
    }
    std::shared_ptr<detail::future_impl> get_needed_future()
    {
        return impl_->f_.get_needed_future();
    }

    void reset()
    {
        promise().swap(*this);
    }

    void swap(promise& other)
    {
        impl_.swap(other.impl_);
        impl_ref_.swap(other.impl_ref_);
    }

private:
    template <typename Y>
    friend class future;
    std::shared_ptr<container_type> impl_;

    // impl_ref_ counts number of promises that refer to container and sets broken_promise
    // exception if there are no more references left.
    boost::intrusive_ptr<container_type> impl_ref_;
};

template <typename T>
struct is_future : std::false_type
{
};

template <typename T>
struct is_future<future<T>> : std::true_type
{
};

template <typename Future, typename Function, typename Promise>
struct assign_future
{
    Future base_future;
    std::decay_t<Function> func;
    Promise promise;

    assign_future(Future future, Function func, Promise promise)
        : base_future(std::move(future)), func(std::move(func)), promise(std::move(promise))
    {
    }

    void operator()()
    {
        try
        {
            set_prom<Function>();
        }
        catch (...)
        {
            promise.set_exception(std::current_exception());
        }
    }

    template <typename F>
    auto set_prom()
    {
        using func_invoke_result = std::invoke_result_t<F, Future&>;
        if constexpr (std::is_same_v<func_invoke_result, void>)
        {
            func(base_future);
            promise.set();
        }
        else if constexpr (is_future<func_invoke_result>::value)
        {
            auto future = func(base_future);
            using proxy_future_type = decltype(future);
            auto pass_value = [](auto future) { return future.get(); };
            using assign_future_type =
                assign_future<proxy_future_type, decltype(pass_value), Promise>;
            future.add_callback(assign_future_type(future, pass_value, promise));
        }
        else
        {
            promise.set(func(base_future));
        }
    }
};

template <typename R>
class future
{
public:
    using value_type = std::decay_t<R>;
    using promise_type = promise<R>;

    // Default constructor will create a future, and will immediately set a
    // broken_promise exception.
    // A default-constructed future is only good for equality assignment to a
    // valid future.
    future() : future(promise_type())
    {
    }

    future(const promise_type& p) : impl_(p.impl_, &p.impl_->f_), value_(p.impl_, &p.impl_->value_)
    {
    }

    future(const future&) = default;

protected:
    // used by future<void> for typeless futures
    future(const std::shared_ptr<detail::future_impl>& impl)
        : impl_(impl), value_() // value should be never used
    {
    }

public:
    future& operator=(const future& t) = default;
    ~future() = default;

    bool has_value() const
    {
        return impl_->has_value();
    }

    bool has_exception() const
    {
        return impl_->has_exception();
    }

    auto get_exception() const
    {
        return impl_->get_exception();
    }

    // queries whether the future contains a value or an exception
    bool ready() const
    {
        return impl_->ready();
    }

    // wait for ready()
    void wait() const
    {
        return impl_->wait();
    }

    template <typename Clock, typename Duration>
    bool timed_wait(time_traits::time_point const& abstime) const
    {
        return impl_->timed_wait(abstime);
    }

    template <typename Rep, typename Period>
    bool timed_wait(time_traits::duration const& reltime) const
    {
        return impl_->timed_wait(reltime);
    }

    // waits for a value, then returns it
    value_type get() const
    {
        return impl_->get(*value_);
    }

    value_type operator()() const
    {
        return get();
    }

    void set_needed() const
    {
        impl_->set_needed();
    }

    template <typename F, typename ThisType = future>
    auto then(F&& func)
    {
        typedef std::invoke_result_t<F, future&> func_return_type;

        if constexpr (is_future<func_return_type>::value)
        {
            using value_type = typename func_return_type::value_type;
            using promise_type = promise<value_type>;
            promise_type prom;
            add_callback(
                assign_future<ThisType, F, promise_type>(*this, std::forward<F>(func), prom));
            return future<value_type>(prom);
        }
        else
        {
            using promise_type = promise<func_return_type>;
            promise_type prom;
            add_callback(
                assign_future<ThisType, F, promise_type>(*this, std::forward<F>(func), prom));
            return future<func_return_type>(prom);
        }
    }

    void add_callback(std::function<void(void)> f)
    {
        return impl_->add_callback(std::move(f));
    }

private:
    template <typename Y>
    friend class future;
    std::shared_ptr<detail::future_impl> impl_;
    std::shared_ptr<value_type> value_;
};

// note, promise<int> must be public for friend to work in specialization (?)
template <>
class promise<void> : public promise<int>
{
private:
    typedef promise<int> base_type;

public:
    promise() : promise<int>()
    {
    }
    promise(const promise& t) : promise<int>(t)
    {
    }
    promise& operator=(const promise& t)
    {
        base_type::operator=(t);
        return *this;
    }
    using base_type::set_exception;
    using base_type::is_needed;
    using base_type::wait_until_needed;
    void set()
    {
        base_type::set(0);
    }
    void set_or_throw()
    {
        base_type::set_or_throw(0);
    }
};

// void specialization, based on Peter Dimov's example
template <>
class future<void> : private future<int>
{
private:
    typedef future<int> base_type;

public:
    using value_type = void;

    future() : base_type()
    {
    }
    future(const future& t) : base_type(static_cast<const future<int>&>(t))
    {
    }

    future(const promise<void>& p) : base_type(static_cast<const promise<int>&>(p))
    {
    }

    future(const std::shared_ptr<detail::future_impl>& impl) : base_type(impl)
    {
    }

    template <typename T>
    future(const future<T>& t) : base_type(t.impl_)
    {
    }

    template <typename T>
    future(const promise<T>& t) : base_type(t.impl_->f_)
    {
    }

    future& operator=(const future& t)
    {
        base_type::operator=(static_cast<const future<int>&>(t));
        return *this;
    }

    template <typename T>
    future& operator=(const future<T>& t)
    {
        future<void> tmp(t);
        base_type::operator=(static_cast<const future<int>&>(tmp));
        return *this;
    }

    using base_type::has_value;
    using base_type::has_exception;
    using base_type::get_exception;
    using base_type::timed_wait;
    using base_type::ready;
    using base_type::wait;
    using base_type::set_needed;
    using base_type::add_callback;

    template <typename F>
    auto then(F&& func)
    {
        return base_type::then<F, future<void>>(std::forward<F>(func));
    }

    void get() const
    {
        wait();
        auto exception = get_exception();
        if (exception) std::rethrow_exception(exception);
    }
};

template <typename R>
class future_wrapper
{
public:
    future_wrapper(const std::function<R(void)>& fn, const promise<R>& ft)
        : fn_(fn), ft_(ft){}; // stores fn and ft

    void operator()() noexcept
    { // executes fn() and places the outcome into ft
        try
        {
            ft_.set(fn_());
        }
        catch (...)
        {
            ft_.set_exception(std::current_exception());
        }
    }
    future<R> get_future() const
    {
        return future<R>(ft_);
    }

private:
    std::function<R(void)> fn_;
    promise<R> ft_;
};

// void specialization
template <>
class future_wrapper<void>
{
public:
    future_wrapper(const std::function<void(void)>& fn, const promise<void>& ft)
        : ft_(ft), fn_(fn){}; // stores fn and ft
    void operator()() noexcept
    { // executes fn() and places the outcome into ft
        try
        {
            fn_();
            ft_.set();
        }
        catch (...)
        {
            ft_.set_exception(std::current_exception());
        }
    }
    future<void> get_future() const
    {
        return future<void>(ft_);
    }

private:
    promise<void> ft_;
    std::function<void(void)> fn_;
};

template <typename T>
future<typename std::decay<T>::type> make_future(T&& value)
{
    promise<typename std::decay<T>::type> ret;
    ret.set(std::forward<T>(value));
    return ret;
}

inline future<void> make_future()
{
    promise<void> prom;
    prom.set();
    return prom;
}

}}
