#pragma once

#include <yplatform/time_traits.h>
#include <boost/thread/mutex.hpp>
#include <boost/thread/condition_variable.hpp>

#include <cassert>
#include <condition_variable>
#include <list>
#include <mutex>
#include <stdlib.h> // abort ()
#include <vector>

// DEBUG
// #include <iostream>

namespace yplatform { namespace future {
namespace detail {
class future_impl;
}

template <typename R>
class future;
template <typename R>
class promise;

namespace detail {

class future_impl
{
public:
    future_impl() : has_value_(false), has_exception_(false), is_needed_()
    {
    }
    bool has_value() const
    {
        std::unique_lock<std::mutex> lck(mutex_);
        return has_value_;
    }
    bool has_exception() const
    {
        std::unique_lock<std::mutex> lck(mutex_);
        return has_exception_;
    }
    std::exception_ptr get_exception() const
    {
        std::unique_lock<std::mutex> lck(mutex_);
        // TODO remove has_exception_ flag because exception_ is bool convertable
        if (!has_exception_) return {};
        return exception_;
    }
    bool ready() const
    {
        std::unique_lock<std::mutex> lck(mutex_);
        return (has_value_ || has_exception_);
    }
    void wait()
    {
        set_needed();
        std::unique_lock<std::mutex> lck(mutex_);
        while (!has_value_ && !has_exception_)
            cond_.wait(lck);
        return;
    }

    bool timed_wait(time_traits::time_point const& abstime)

    {
        set_needed();
        std::unique_lock<std::mutex> lck(mutex_);
        while (!has_value_ && !has_exception_)
            if (cond_.wait_until(lck, abstime) == std::cv_status::timeout)
                return false; /* timeout */
        return true;
    }

    bool timed_wait(time_traits::duration const& reltime)
    {
        set_needed();
        std::unique_lock<std::mutex> lck(mutex_);
        while (!has_value_ && !has_exception_)
            if (cond_.wait_for(lck, reltime) == std::cv_status::timeout) return false; /* timeout */
        return true;
    }

    // Could return by-ref if set_value only called once
    template <typename R>
    R get(const R& value)
    {
        set_needed();
        std::unique_lock<std::mutex> lck(mutex_);
        while (!has_value_ && !has_exception_)
            cond_.wait(lck);
        if (has_exception_) std::rethrow_exception(exception_);
        return value;
    }

    template <typename R>
    bool set_value(R&& r, R& value)
    {
        std::unique_lock<std::mutex> lck(mutex_);
        if (has_value_ || has_exception_) return false;
        value = std::forward<R>(r);
        has_value_ = true;
        notify(lck);
        return true;
    }

    void set_exception(std::exception_ptr e)
    {
        std::unique_lock<std::mutex> lck(mutex_);
        if (has_value_ || has_exception_) return;
        exception_ = std::move(e);
        has_exception_ = true;
        notify(lck);
    }

    void add_callback(const std::function<void(void)>& f)
    {

        std::unique_lock<std::mutex> lck(mutex_);
        if (has_value_ || has_exception_)
        {
            lck.unlock(); // never call a callback within the mutex
            f();          // future already fulfilled.  Call the callback immediately.
            return;
        }
        assert(!callback_ && "only one future callback is allowed");
        callback_ = f;
    }

    void add_callback(std::function<void(void)>&& f)
    {

        std::unique_lock<std::mutex> lck(mutex_);
        if (has_value_ || has_exception_)
        {
            lck.unlock(); // never call a callback within the mutex
            f();          // future already fulfilled.  Call the callback immediately.
            return;
        }
        assert(!callback_ && "only one future callback is allowed");
        callback_ = std::move(f);
    }

    std::shared_ptr<future_impl> get_needed_future() const
    {
        std::unique_lock<std::mutex> lck(mutex_);
        if (!is_needed_) // allocate if desired
            is_needed_.reset(new future_impl);
        return is_needed_;
    }

    // as-needed functionality permits lazy eval and as-needed producer/consumer
    void set_needed()
    {
        std::shared_ptr<future_impl> n = get_needed_future();
        n->set();
    }

    bool is_needed() const
    {
        std::unique_lock<std::mutex> lck(mutex_);
        // if we are bound, we always say we are already needed
        return ((is_needed_ && is_needed_->ready()) || has_value_ || has_exception_);
    }
    void wait_until_needed() const
    {
        std::shared_ptr<future_impl> n = get_needed_future();
        n->wait();
    }

private:
    void notify(std::unique_lock<std::mutex>& lck)
    {
        cond_.notify_all();
        std::function<void(void)> cb;
        cb.swap(callback_);
        lck.unlock();
        if (cb) cb();
    }
    bool set()
    { // a very simple set, used for as_needed_ future
        std::unique_lock<std::mutex> lck(mutex_);
        if (has_value_ || has_exception_) return false;
        has_value_ = true;
        notify(lck);
        return true;
    }
    bool has_value_;
    bool has_exception_;
    std::exception_ptr exception_;
    mutable std::mutex mutex_;
    mutable std::condition_variable_any cond_;
    mutable std::shared_ptr<future_impl> is_needed_;
    std::function<void(void)> callback_;
};

template <typename R>
class promise_impl
{
    template <typename T>
    struct underlying
    {
        using type = std::decay_t<T>;
    };

    template <>
    struct underlying<void>
    {
        using type = int;
    };

    template <typename T>
    using underlying_t = typename underlying<T>::type;

public:
    using value_type = underlying_t<R>;

    detail::future_impl f_;
    value_type value_;
    std::atomic_int_least32_t references_ = { 0 };
};

template <typename R>
inline void intrusive_ptr_add_ref(promise_impl<R>* p)
{
    ++p->references_;
}

template <typename R>
inline void intrusive_ptr_release(promise_impl<R>* p)
{
    if ((--p->references_) == 0 && !p->f_.ready())
    {
        try
        {
            p->f_.set_exception(std::make_exception_ptr(broken_promise()));
        }
        catch (...)
        {
        }
    }
}

} // namespace detail
} // namespace future
}
