#pragma once

#include <yplatform/future/future.hpp>
#include <boost/thread/mutex.hpp>

namespace yplatform { namespace future {

typedef future<void> future_void_t;
typedef promise<void> promise_void_t;

namespace detail {

class future_multi_and_impl : ::boost::noncopyable
{
public:
    future_multi_and_impl(std::size_t total) : total_(total), error_set_(false)
    {
        if (total == 0)
        {
            p_.set();
        }
    }
    template <typename Future>
    void operator()(Future f)
    {
        std::lock_guard<std::mutex> lck(mutex_);
        counter_++;
        assert(counter_ <= total_);
        if (error_set_) return;
        try
        {
            f.get();
            if (counter_ == total_) p_.set();
        }
        catch (...)
        {
            p_.set_current_exception();
            error_set_ = true;
        }
    }

    promise_void_t p_;
    std::size_t counter_ = 0;
    std::size_t total_;
    bool error_set_;
    std::mutex mutex_;
};

template <typename Implementation>
class future_multi_func
{
public:
    future_multi_func(std::size_t total) : impl_(new Implementation(total))
    {
    }
    template <typename Future>
    void operator()(Future f)
    {
        (*impl_)(f);
    }
    promise_void_t get_promise()
    {
        return impl_->p_;
    }

protected:
    boost::shared_ptr<Implementation> impl_;
};

template <typename Collection, typename Implementation, typename Getter>
future_void_t future_multi_base(const Collection& futures, Getter&& get)
{
    future_multi_func<Implementation> fres(futures.size());
    for (auto& item : futures)
    {
        auto f = get(item);
        f.add_callback([fres, f]() mutable { fres(f); });
    }
    return fres.get_promise();
}

template <typename Collection, typename Implementation>
future_void_t future_multi_base(const Collection& futures)
{
    future_multi_func<Implementation> fres(futures.size());
    for (future_void_t f : futures)
    {
        f.add_callback([fres, f]() mutable { fres(f); });
    }
    return fres.get_promise();
}
}

template <typename Collection, typename Getter>
future_void_t future_multi_and(const Collection& futures, Getter&& get)
{
    return detail::future_multi_base<Collection, detail::future_multi_and_impl, Getter>(
        futures, std::forward<Getter>(get));
}

template <typename Collection>
future_void_t future_multi_and(const Collection& futures)
{
    return detail::future_multi_base<Collection, detail::future_multi_and_impl>(futures);
}

}}
