#pragma once

#include <yplatform/config.h>
#include <rapidjson/document.h>
#include <rapidjson/writer.h>
#include <rapidjson/prettywriter.h>
#include <rapidjson/error/en.h>
#include <boost/range/algorithm.hpp>
#include <optional>
#include <sstream>

namespace yplatform {

namespace detail {

// Helper for zero copy json stringify.
class string_holder
{
public:
    typedef char Ch;
    string_holder(string& s) : s_(s)
    {
        s_.reserve(4096);
    }
    void Put(char c)
    {
        s_.push_back(c);
    }
    void Clear()
    {
        s_.clear();
    }
    void Flush()
    {
        return;
    }
    size_t Size() const
    {
        return s_.length();
    }

private:
    string& s_;
};

// Is used as a base class for json_value to init document before json_value_ref.
class json_document
{
protected:
    typedef rapidjson::Document implementation_type;
    implementation_type::AllocatorType alloc_{ 256 };
    implementation_type document_{ &alloc_ };
};

template <class T>
struct dependent_false : std::false_type
{
};
}

enum class json_type : int
{
    tnull = rapidjson::kNullType,
    tbool = rapidjson::kTrueType,
    tobject = rapidjson::kObjectType,
    tarray = rapidjson::kArrayType,
    tstring = rapidjson::kStringType,
    tnumber = rapidjson::kNumberType
};

template <typename WrapperType, typename IteratorType>
class json_array_iterator : public std::iterator<std::forward_iterator_tag, WrapperType>
{
public:
    typedef WrapperType wrapper_type;
    typedef typename WrapperType::allocator_type allocator_type;
    typedef IteratorType iterator_type;

    json_array_iterator(iterator_type impl, allocator_type& alloc) : impl_(impl), alloc_(&alloc)
    {
    }

    wrapper_type operator*() const
    {
        return std::remove_const_t<wrapper_type>(*impl_, *alloc_);
    }

    auto operator-> () const
    {
        return std::remove_const_t<wrapper_type>(*impl_, *alloc_);
    }

    json_array_iterator& operator++()
    {
        ++impl_;
        return *this;
    }

    json_array_iterator operator++(int)
    {
        json_array_iterator copy = *this;
        impl_++;
        return copy;
    }

    bool operator==(const json_array_iterator& other) const
    {
        return impl_ == other.impl_;
    }

    bool operator!=(const json_array_iterator& other) const
    {
        return impl_ != other.impl_;
    }

private:
    iterator_type impl_;
    allocator_type* alloc_;
};

template <typename WrapperType, typename IteratorType>
class json_object_iterator : public std::iterator<std::forward_iterator_tag, WrapperType>
{
public:
    typedef WrapperType wrapper_type;
    typedef typename WrapperType::allocator_type allocator_type;
    typedef IteratorType iterator_type;

    json_object_iterator(iterator_type impl, allocator_type& alloc) : impl_(impl), alloc_(&alloc)
    {
    }

    auto key() const
    {
        return string_view(impl_->name.GetString(), impl_->name.GetStringLength());
    }

    wrapper_type operator*() const
    {
        return std::remove_const_t<wrapper_type>(impl_->value, *alloc_);
    }

    json_object_iterator& operator++()
    {
        ++impl_;
        return *this;
    }

    json_object_iterator operator++(int)
    {
        json_object_iterator copy = *this;
        impl_++;
        return copy;
    }

    auto operator==(const json_object_iterator& other) const
    {
        return impl_ == other.impl_;
    }

    auto operator!=(const json_object_iterator& other) const
    {
        return impl_ != other.impl_;
    }

private:
    iterator_type impl_;
    allocator_type* alloc_;
};

template <typename Value>
class basic_json_value_ref
{
public:
    typedef rapidjson::Value value_type;
    typedef Value implementation_type;
    typedef rapidjson::Document::AllocatorType allocator_type;
    typedef basic_json_value_ref<implementation_type> this_type;
    typedef rapidjson::Value::StringRefType string_ref_type;

    const this_type& cref;

    basic_json_value_ref(implementation_type& impl, allocator_type& alloc)
        : cref(*this), pimpl_(&impl), alloc_(&alloc)
    {
    }

    basic_json_value_ref& operator=(const basic_json_value_ref& t)
    {
        if (&t != this)
        {
            assign_copy(t);
        }
        return *this;
    }

    basic_json_value_ref& operator=(basic_json_value_ref&& t)
    {
        if (&t != this)
        {
            swap(t);
        }
        return *this;
    }

    basic_json_value_ref& operator=(const string& s)
    {
        pimpl_->SetString(s.data(), s.size(), allocator()); // make a copy
        return *this;
    }

    basic_json_value_ref& operator=(const string_view& s)
    {
        pimpl_->SetString(s.data(), s.size(), allocator()); // make a copy
        return *this;
    }

    basic_json_value_ref& operator=(const char* s)
    {
        pimpl_->SetString(s, strlen(s)); // store as a reference
        return *this;
    }

    template <typename T, typename = std::enable_if_t<std::is_arithmetic_v<T>>>
    basic_json_value_ref& operator=(T t)
    {
        pimpl_->operator=(t);
        return *this;
    }

    void swap(basic_json_value_ref& other)
    {
        if (&other.allocator() != &allocator())
        {
            assign_copy(other);
        }
        else
        {
            pimpl_->Swap(*other.pimpl_);
        }
    }

    void assign_copy(const basic_json_value_ref& other)
    {
        value_type copy(other.native(), allocator());
        pimpl_->Swap(copy);
    }

    template <typename T>
    bool operator==(const T& t) const
    {
        if constexpr (std::is_base_of_v<this_type, std::decay_t<T>>)
        {
            return *pimpl_ == *t.pimpl_;
        }
        else if constexpr (std::is_same_v<std::string, std::decay_t<T>>)
        {
            return *pimpl_ == string_ref_type(t.data(), t.size());
        }
        else
        {
            return *pimpl_ == t;
        }
    }

    template <typename T>
    bool operator!=(const T& t) const
    {
        if constexpr (std::is_base_of_v<this_type, std::decay_t<T>>)
        {
            return *pimpl_ != *t.pimpl_;
        }
        else if constexpr (std::is_same_v<std::string, std::decay_t<T>>)
        {
            return *pimpl_ != string_ref_type(t.data(), t.size());
        }
        else
        {
            return *pimpl_ != t;
        }
    }

    this_type push_back()
    {
        if (is_null()) set_array();
        if (!is_array()) throw std::runtime_error("not a json array");
        pimpl_->PushBack(value_type(), allocator());
        return this_type(pimpl_->operator[](size() - 1), allocator());
    }

    // Transfer ownership of the referenced value.
    template <typename T>
    void push_back(T&& t)
    {
        if (is_null()) set_array();
        if (!is_array()) throw std::runtime_error("not a json array");

        if constexpr (std::is_base_of_v<this_type, std::remove_reference_t<T>>)
        {
            // Can't transfer ownership if allocators are different.
            if (&t.allocator() != &allocator())
            {
                value_type copy(*t.pimpl_, allocator());
                pimpl_->PushBack(std::move(copy), allocator());
            }
            else
            {
                pimpl_->PushBack(std::move(*t.pimpl_), allocator());
            }
        }
        else
        {
            if constexpr (std::is_arithmetic_v<std::remove_reference_t<T>>)
            {
                pimpl_->PushBack(value_type(t), allocator());
            }
            else if constexpr (std::is_same_v<std::string, std::decay_t<T>>)
            {
                pimpl_->PushBack(value_type(t.data(), t.size(), allocator()), allocator());
            }
            else if constexpr (std::is_same_v<char*, std::decay_t<T>>)
            {
                pimpl_->PushBack(value_type(t), allocator());
            }
            else if constexpr (std::is_same_v<char const*, std::decay_t<T>>)
            {
                pimpl_->PushBack(value_type(t), allocator());
            }
            else
            {
                static_assert(
                    detail::dependent_false<T>::value,
                    "unsupported type for json_value::push_back()");
            }
        }
    }

    operator bool() const
    {
        return !is_null();
    }

    bool isMember(const char* key) const
    {
        return is_object() && pimpl_->HasMember(key);
    }

    bool isMember(const string& key) const
    {
        string_ref_type key_ref(key.data(), key.size());
        return is_object() && pimpl_->HasMember(key_ref);
    }

    bool isMember(const string_view& key) const
    {
        string_ref_type key_ref(key.data(), key.size());
        return is_object() && pimpl_->HasMember(key_ref);
    }

    bool has_member(const char* key) const
    {
        return isMember(key);
    }
    bool has_member(const string& key) const
    {
        return isMember(key);
    }
    bool has_member(const string_view& key) const
    {
        return isMember(key);
    }

    template <typename Key>
    bool removeMember(Key&& key)
    {
        return remove_member(std::forward<Key>(key));
    }

    bool remove_member(const char* key)
    {
        return is_object() && pimpl_->RemoveMember(key);
    }

    bool remove_member(const string& key)
    {
        value_type native_key(rapidjson::StringRef(key.data(), key.size()));
        return is_object() && pimpl_->RemoveMember(native_key);
    }

    void set_object()
    {
        pimpl_->SetObject();
    }

    void set_array()
    {
        pimpl_->SetArray();
    }

    bool is_string() const
    {
        return pimpl_->IsString();
    }
    bool is_null() const
    {
        return pimpl_->IsNull();
    }
    bool is_bool() const
    {
        return pimpl_->IsBool();
    }
    bool is_object() const
    {
        return pimpl_->IsObject();
    }
    bool is_array() const
    {
        return pimpl_->IsArray();
    }
    bool is_number() const
    {
        return pimpl_->IsNumber();
    }
    bool is_int() const
    {
        return pimpl_->IsInt();
    }
    bool is_uint() const
    {
        return pimpl_->IsUint();
    }
    bool is_int64() const
    {
        return pimpl_->IsInt64();
    }
    bool is_uint64() const
    {
        return pimpl_->IsUint64();
    }
    bool is_double() const
    {
        return pimpl_->IsDouble();
    }

    auto to_int64() const
    {
        if (!is_int64()) throw std::runtime_error("invalid json type (expected int64)");
        return pimpl_->GetInt64();
    }
    auto to_uint64() const
    {
        if (!is_uint64()) throw std::runtime_error("invalid json type (expected uint64)");
        return pimpl_->GetUint64();
    }
    auto to_double() const
    {
        if (!is_number()) throw std::runtime_error("invalid json type (expected double)");
        return pimpl_->GetDouble();
    }
    auto to_bool() const
    {
        if (!is_bool()) throw std::runtime_error("invalid json type (expected bool)");
        return pimpl_->GetBool();
    }

    bool empty() const
    {
        return size() == 0;
    }

    size_t size() const
    {
        switch (type())
        {
        case json_type::tarray:
            return pimpl_->Size();
        case json_type::tobject:
            return pimpl_->MemberCount();
        default:
            return 0;
        }
    }

    json_type type() const
    {
        auto native = pimpl_->GetType();
        if (native == rapidjson::kTrueType || native == rapidjson::kFalseType)
        {
            return json_type::tbool;
        }
        return static_cast<json_type>(native);
    }

    const this_type operator[](size_t index) const
    {
        if (!is_array() || index >= size())
        {
            static allocator_type fake_allocator;
            static value_type value;
            return this_type(value, fake_allocator);
        }
        return this_type(pimpl_->operator[](index), allocator());
    }

    const this_type operator[](const char* key) const
    {
        return find(value_type(string_ref_type(key, strlen(key))));
    }

    const this_type operator[](const string& key) const
    {
        return find(value_type(string_ref_type(key.begin(), key.size())));
    }

    const this_type operator[](const string_view& key) const
    {
        return find(value_type(string_ref_type(key.begin(), key.size())));
    }

    this_type operator[](size_t index)
    {
        if (!is_array()) throw std::runtime_error("invalid json type");
        return this_type(pimpl_->operator[](index), allocator());
    }

    this_type operator[](const char* key)
    {
        return find(value_type(string_ref_type(key, strlen(key))));
    }

    this_type operator[](const string& key)
    {
        return find(value_type(key.begin(), key.size(), allocator()));
    }

    this_type operator[](const string_view& key)
    {
        return find(value_type(key.begin(), key.size(), allocator()));
    }

    string to_string() const
    {
        string ret;
        switch (type())
        {
        case json_type::tnull:
            return "";
        case json_type::tbool:
            return to_bool() ? "true" : "false";
        case json_type::tnumber:
            if (is_int64())
            {
                return std::to_string(to_int64());
            }
            else if (is_uint64())
            {
                return std::to_string(to_uint64());
            }
            else if (is_double())
            {
                return std::to_string(to_double());
            }
        default:
            if (!is_string()) throw std::runtime_error("not a string-convertable value");
            return string(pimpl_->GetString(), pimpl_->GetStringLength());
        };
    }

    string to_string(const string& default_value) const
    {
        string ret;
        switch (type())
        {
        case json_type::tnull:
            return default_value;
        case json_type::tbool:
            return to_bool() ? "true" : "false";
        case json_type::tnumber:
            if (is_int64())
            {
                return std::to_string(to_int64());
            }
            else if (is_uint64())
            {
                return std::to_string(to_uint64());
            }
            else if (is_double())
            {
                return std::to_string(to_double());
            }
        default:
            if (!is_string()) return default_value;
            return string(pimpl_->GetString(), pimpl_->GetStringLength());
        };
    }

    auto to_string_view(bool empty_if_missing = false) const
    {
        bool return_empty = is_null() || (!is_string() && empty_if_missing);
        if (return_empty)
        {
            static const string null_str = "";
            return string_view(null_str);
        }
        if (!is_string()) throw std::runtime_error("not a string value");
        return string_view(pimpl_->GetString(), pimpl_->GetStringLength());
    }

    template <typename T, typename = std::enable_if_t<std::is_pod_v<T>>>
    T get(T default_value) const
    {
        return (pimpl_->template Is<T>()) ? pimpl_->template Get<T>() : default_value;
    }

    auto members_begin() const
    {
        if (!is_object()) throw std::runtime_error("invalid json type");
        return json_object_iterator<this_type, value_type::MemberIterator>(
            pimpl_->MemberBegin(), allocator());
    }

    auto members_end() const
    {
        if (!is_object()) throw std::runtime_error("invalid json type");
        return json_object_iterator<this_type, value_type::MemberIterator>(
            pimpl_->MemberEnd(), allocator());
    }

    auto array_begin() const
    {
        if (!is_array()) throw std::runtime_error("invalid json type");
        return json_array_iterator<this_type, value_type::ValueIterator>(
            pimpl_->Begin(), allocator());
    }

    auto array_end() const
    {
        if (!is_array()) throw std::runtime_error("invalid json type");
        return json_array_iterator<this_type, value_type::ValueIterator>(
            pimpl_->End(), allocator());
    }

    auto members_begin()
    {
        if (!is_object()) throw std::runtime_error("invalid json type");
        return json_object_iterator<this_type, value_type::MemberIterator>(
            pimpl_->MemberBegin(), allocator());
    }

    auto members_end()
    {
        if (!is_object()) throw std::runtime_error("invalid json type");
        return json_object_iterator<this_type, value_type::MemberIterator>(
            pimpl_->MemberEnd(), allocator());
    }

    auto array_begin()
    {
        if (!is_array()) throw std::runtime_error("invalid json type");
        return json_array_iterator<this_type, value_type::ValueIterator>(
            pimpl_->Begin(), allocator());
    }

    auto array_end()
    {
        if (!is_array()) throw std::runtime_error("invalid json type");
        return json_array_iterator<this_type, value_type::ValueIterator>(
            pimpl_->End(), allocator());
    }

    struct array_items_wrapper
    {
        auto begin()
        {
            if (!pimpl_->IsArray()) throw std::runtime_error("invalid json type");
            return json_array_iterator<this_type, value_type::ValueIterator>(
                pimpl_->Begin(), *alloc_);
        }

        auto end()
        {
            if (!pimpl_->IsArray()) throw std::runtime_error("invalid json type");
            return json_array_iterator<this_type, value_type::ValueIterator>(
                pimpl_->End(), *alloc_);
        }
        implementation_type* pimpl_;
        allocator_type* alloc_;
    };

    struct const_array_items_wrapper
    {
        auto begin() const
        {
            if (!pimpl_->IsArray()) throw std::runtime_error("invalid json type");
            return json_array_iterator<this_type, value_type::ValueIterator>(
                pimpl_->Begin(), *alloc_);
        }

        auto end() const
        {
            if (!pimpl_->IsArray()) throw std::runtime_error("invalid json type");
            return json_array_iterator<this_type, value_type::ValueIterator>(
                pimpl_->End(), *alloc_);
        }
        implementation_type* pimpl_;
        allocator_type* alloc_;
    };

    array_items_wrapper array_items()
    {
        return array_items_wrapper{ pimpl_, alloc_ };
    }

    const_array_items_wrapper array_items() const
    {
        return const_array_items_wrapper{ pimpl_, alloc_ };
    }

    implementation_type& native() const
    {
        return *pimpl_;
    }

    allocator_type& allocator() const
    {
        return *alloc_;
    }

    string stringify() const
    {
        if (is_null()) return "";
        string ret;
        detail::string_holder s(ret);
        rapidjson::Writer<detail::string_holder> writer(s);
        pimpl_->Accept(writer);
        return ret;
    }

    string pretty_stringify() const
    {
        if (is_null()) return "";
        string ret;
        detail::string_holder s(ret);
        rapidjson::PrettyWriter<detail::string_holder> writer(s);
        pimpl_->Accept(writer);
        return ret;
    }

private:
    template <typename Key>
    const this_type find(const Key& key) const
    {
        if (is_object())
        {
            auto it = pimpl_->FindMember(key);
            if (it != pimpl_->MemberEnd()) return this_type(it->value, allocator());
        }
        static allocator_type fake_allocator;
        static value_type value;
        return this_type(value, fake_allocator);
    }

    template <typename Key>
    this_type find(const Key& key)
    {
        if (is_null()) set_object();
        if (!is_object()) throw std::runtime_error("not an object");

        auto it = pimpl_->FindMember(key);
        if (it != pimpl_->MemberEnd()) return this_type(it->value, allocator());
        pimpl_->AddMember(value_type(key, allocator()), value_type(), allocator());
        return this_type(pimpl_->operator[](key), allocator());
    }

    implementation_type* pimpl_;
    allocator_type* alloc_;
};

template <typename T>
inline string to_string(
    const basic_json_value_ref<T>& json_ref,
    const string& default_value = string())
{
    switch (json_ref.type())
    {
    case json_type::tstring:
        return json_ref.to_string();
    default:
        return default_value;
    };
}

typedef basic_json_value_ref<rapidjson::Value> json_value_ref;

class json_value
    : public detail::json_document
    , public json_value_ref
{
public:
    typedef detail::json_document::implementation_type implementation_type;

    json_value() : json_document(), json_value_ref(document_, document_.GetAllocator())
    {
    }

    json_value(const json_value& other) : json_value()
    {
        json_value_ref::assign_copy(other);
    }

    template <typename T>
    json_value(const basic_json_value_ref<T>& ref) : json_value()
    {
        operator=(ref);
    }

    json_value(int value) : json_value()
    {
        operator=(value);
    }
    json_value(int64_t value) : json_value()
    {
        operator=(value);
    }
    json_value(uint64_t value) : json_value()
    {
        operator=(value);
    }
    json_value(const char* value) : json_value()
    {
        operator=(value);
    }
    json_value(string&& value) : json_value()
    {
        operator=(std::move(value));
    }

    json_value(json_type type) : json_value()
    {
        switch (type)
        {
        case json_type::tstring:
            operator=(std::string());
            break;
        case json_type::tobject:
            set_object();
            break;
        case json_type::tarray:
            set_array();
            break;
        case json_type::tnumber:
            operator=(0);
            break;
        case json_type::tbool:
            operator=(false);
            break;
        default:
            break;
        }
    }

    json_value& operator=(const json_value& other)
    {
        if (this != &other)
        {
            json_value_ref::assign_copy(other);
        }
        return *this;
    }

    json_value& operator=(const json_value_ref& ref)
    {
        assign_copy(ref);
        return *this;
    }

    using json_value_ref::operator=;

    std::optional<string> parse(const string& src)
    {
        return parse(src.data(), src.size());
    }

    std::optional<string> parse(const string& src, json_type expected_type)
    {
        auto error = parse(src);
        if (!error && type() != expected_type)
        {
            return "unexpected json document type";
        }
        return error;
    }

    std::optional<string> parse(const char* data, size_t len)
    {
        if (len == 0) return {};
        rapidjson::ParseResult result = document_.Parse(data, len);
        if (!result)
        {
            std::stringstream ss;
            ss << "JSON parse error: " << rapidjson::GetParseError_En(result.Code()) << " "
               << result.Offset();
            return ss.str();
        }
        return {};
    }
};

} // namespace yplatform