#pragma once

#include <yplatform/ptree.h>
#include <yplatform/api.h>
#include <yplatform/log/contains_logger.h>
#include <yplatform/log/service.h>
#include <yplatform/log/typed.h>
#include <yplatform/log/logger.h>
#include <yplatform/log/find.h>
#include <yplatform/application/task_context.h>

#include <boost/optional.hpp>
#include <boost/asio/io_service.hpp>

namespace yplatform { namespace log {

void log_init(const ptree& cfg);
void log_load_cfg(const ptree& cfg);

void global_log_init(
    const std::vector<std::shared_ptr<spdlog::logger>>& loggers,
    const std::vector<spdlog::sink_ptr>& sinks);
void global_log_init(const settings&);
void global_log_init(const ptree&);
void global_log_deinit();
void init_global_log_console();
void init_global_log_file(const std::string& path);

void init_console(boost::asio::io_service&);
void init_file(boost::asio::io_service&, const std::string& path);

namespace detail {

void global_log_init(std::shared_ptr<detail::service> service);

struct enable_type
{
};

template <typename From, typename To, typename Ret = enable_type>
struct enable_if_same_or_derived
    : boost::enable_if<boost::is_base_of<From, typename boost::remove_reference<To>::type>, Ret>
{
};

template <typename From, typename To, typename Ret = enable_type>
struct disable_if_same_or_derived
    : boost::disable_if<boost::is_base_of<From, typename boost::remove_reference<To>::type>, Ret>
{
};

template <typename T>
inline decltype(auto) get_logger_from(T&& t)
{
    if constexpr (
        std::is_same_v<source, std::decay_t<T>> || std::is_same_v<typed::logger, std::decay_t<T>>)
    {
        return std::forward<T>(t);
    }
    else
    {
        try
        {
            if (t) return t->logger();
        }
        catch (...)
        {
        }
        return YGLOBAL_LOGGER;
    }
}

template <typename T>
inline const string& get_uniq_id(T&& t)
{
    static const string empty;
    if constexpr (
        std::is_same_v<source, std::decay_t<T>> || std::is_same_v<typed::logger, std::decay_t<T>> ||
        std::is_same_v<std::nullptr_t, std::decay_t<T>>)
    {
        return empty;
    }
    else if constexpr (std::is_same_v<std::string, std::decay_t<T>>)
    {
        return t;
    }
    else
    {
        return t ? t->uniq_id() : empty;
    }
}
template <typename T, typename Y>
inline const string& get_uniq_id(T&& /*t*/, Y&& y)
{
    return get_uniq_id(std::forward<Y>(y));
}

template <typename This>
source& get_logger(
    This& t,
    typename enable_if_same_or_derived<::yplatform::log::contains_logger, This>::type* = 0) noexcept
{
    try
    {
        return t.logger();
    }
    catch (...)
    {
        return YGLOBAL_LOGGER;
    }
}

template <typename This>
source& get_logger(
    const This& t,
    typename enable_if_same_or_derived<::yplatform::log::contains_logger, This>::type* = 0) noexcept
{
    try
    {
        return t.logger();
    }
    catch (...)
    {
        return YGLOBAL_LOGGER;
    }
}

template <typename This>
source& get_logger(
    This& /*t*/,
    typename disable_if_same_or_derived<::yplatform::log::contains_logger, This>::type* =
        0) noexcept
{
    return YGLOBAL_LOGGER;
}

template <typename This>
source& get_logger(
    const This& /*t*/,
    typename disable_if_same_or_derived<::yplatform::log::contains_logger, This>::type* =
        0) noexcept
{
    return YGLOBAL_LOGGER;
}

template <typename Logger>
class logger_noexcept_wrapper
{
public:
    class stream
    {
    public:
        stream() noexcept
        {
        }
        // Exceptions thrown here will be caught in write().
        stream(typename Logger::stream&& impl) : impl_(std::forward<typename Logger::stream>(impl))
        {
        }

        template <class T>
        stream& operator<<(const T& value) noexcept
        {
            try
            {
                if (impl_) impl_.get() << value;
            }
            catch (...)
            {
            }
            return *this;
        }

        template <class T>
        stream& operator<<(T&& value) noexcept
        {
            try
            {
                if (impl_) impl_.get() << std::forward<T>(value);
            }
            catch (...)
            {
            }
            return *this;
        }

        operator bool() const noexcept
        {
            try
            {
                return impl_ ? bool(impl_.get()) : false;
            }
            catch (...)
            {
                return false;
            }
        }

        void close() noexcept
        {
            try
            {
                if (impl_) impl_.get().close();
            }
            catch (...)
            {
            }
        }

    private:
        boost::optional<typename Logger::stream> impl_;
    };

    logger_noexcept_wrapper(Logger& logger) noexcept : logger(logger)
    {
    }

    stream write(yplatform::log::severity_level severity) noexcept
    {
        try
        {
            return { logger.write(severity) };
        }
        catch (...)
        {
            return {};
        }
    }

private:
    Logger& logger;
};

template <typename Logger>
logger_noexcept_wrapper<Logger> make_noexcept_wrapper(Logger& l) noexcept
{
    return logger_noexcept_wrapper<Logger>(l);
}

} // namespace detail

#define __FIRST_ARG(arg, ...) arg

#define YLOG(logger, severity)                                                                     \
    for (auto __stream = yplatform::log::detail::make_noexcept_wrapper(logger).write(              \
             yplatform::log::severity_level::severity);                                            \
         __stream;                                                                                 \
         __stream.close())                                                                         \
    __stream

#define YLOG_CTX(logger, context, severity)                                                        \
    YLOG(logger, severity) << yplatform::log::detail::get_uniq_id(context) << " "

// Preferred - context specific logging. Use with task or module contexts.
#define LDEBUG_(...)                                                                               \
    YLOG_CTX(                                                                                      \
        ::yplatform::log::detail::get_logger_from(__FIRST_ARG(__VA_ARGS__)),                       \
        ::yplatform::log::detail::get_uniq_id(__VA_ARGS__),                                        \
        debug)
#define LINFO_(...)                                                                                \
    YLOG_CTX(                                                                                      \
        ::yplatform::log::detail::get_logger_from(__FIRST_ARG(__VA_ARGS__)),                       \
        ::yplatform::log::detail::get_uniq_id(__VA_ARGS__),                                        \
        info)
#define LWARN_(...)                                                                                \
    YLOG_CTX(                                                                                      \
        ::yplatform::log::detail::get_logger_from(__FIRST_ARG(__VA_ARGS__)),                       \
        ::yplatform::log::detail::get_uniq_id(__VA_ARGS__),                                        \
        info)
#define LERR_(...)                                                                                 \
    YLOG_CTX(                                                                                      \
        ::yplatform::log::detail::get_logger_from(__FIRST_ARG(__VA_ARGS__)),                       \
        ::yplatform::log::detail::get_uniq_id(__VA_ARGS__),                                        \
        error)
#define LALERT_(...)                                                                               \
    YLOG_CTX(                                                                                      \
        ::yplatform::log::detail::get_logger_from(__FIRST_ARG(__VA_ARGS__)),                       \
        ::yplatform::log::detail::get_uniq_id(__VA_ARGS__),                                        \
        alert)
#define LEMERG_(...)                                                                               \
    YLOG_CTX(                                                                                      \
        ::yplatform::log::detail::get_logger_from(__FIRST_ARG(__VA_ARGS__)),                       \
        ::yplatform::log::detail::get_uniq_id(__VA_ARGS__),                                        \
        emerg)

// For other purposes.
#define L_(severity) YLOG(YGLOBAL_LOGGER, severity)

// ----------------------- DEPRECATED -------------------------
#define YLOG_LOCAL(severity) YLOG(::yplatform::log::detail::get_logger(*this), severity)
#define YLOG_GLOBAL(severity) YLOG(YGLOBAL_LOGGER, severity)

#define YLOG_CTX_LOCAL(context, severity)                                                          \
    YLOG_CTX(::yplatform::log::detail::get_logger(*this), context, severity)

#define YLOG_CTX_GLOBAL(context, severity) YLOG_CTX(YGLOBAL_LOGGER, context, severity)

// Shorter
#define YLOG_L(severity) YLOG_LOCAL(severity)
#define YLOG_G(severity) YLOG_GLOBAL(severity)

// Switchable DEBUG macros(s)
#ifdef DEBUG

#define YDEBUG_LOG(logger, args) YLOG(logger, debug) << args
#define YDEBUG_LOCAL(args) YLOG_L(debug) << args
#define YDEBUG_GLOBAL(args) YLOG_G(debug) << args
// Context
#define YDEBUG_CTX(logger, context, args) YLOG_CTX(logger, context, debug) << args
#define YDEBUG_CTX_LOCAL(context, args) YLOG_CTX_LOCAL(context, debug) << args
#define YDEBUG_CTX_GLOBAL(context, args) YLOG_CTX_GLOBAL(context, debug) << args
// Shorter
#define YDEBUG_L(args) YDEBUG_LOCAL(args)
#define YDEBUG_G(args) YDEBUG_LOCAL(args)

#else

#define YDEBUG_LOG(logger, args)
#define YDEBUG_LOCAL(args)
#define YDEBUG_GLOBAL(args)
// Context
#define YDEBUG_CTX(logger, context, args)
#define YDEBUG_CTX_LOCAL(context, args)
#define YDEBUG_CTX_GLOBAL(context, args)
// Shorter
#define YDEBUG_L(args)
#define YDEBUG_G(args)

#endif

// Switchable INFO macros(s)
#define YINFO(logger, args) YLOG(logger, info) << args
#define YINFO_LOCAL(args) YLOG_L(info) << args
#define YINFO_GLOBAL(args) YLOG_G(info) << args
// Context
#define YINFO_CTX(logger, context, args) YLOG_CTX(logger, context, info) << args
#define YINFO_CTX_LOCAL(logger, context, args) YLOG_CTX_LOCAL(context, info) << args
#define YINFO_CTX_GLOBAL(logger, context, args) YLOG_CTX_GLOBAL(context, info) << args
// Shorter
#define YINFO_L(args) YINFO_LOCAL(args)
#define YINFO_G(args) YINFO_LOCAL(args)

#define L_SEV(severity) YLOG(YGLOBAL_LOGGER, severity)
#define TASK_LOG(context, severity)                                                                \
    YLOG(::yplatform::log::detail::get_logger_from(context), severity)
// ----------------------- DEPRECATED -------------------------
}}
