#pragma once

#include <yplatform/log/detail/prepare_config.h>
#include <yplatform/log/reopenable.h>
#include <yplatform/log/typed_stream.h>
#include <yplatform/log/severity_level.h>

#include <yplatform/log/detail/spd.h>
#include <yplatform/log/spdlog_stream.h>
#include <spdlog/sinks/null_sink.h>

#include <cassert>
#include <unordered_map>

namespace yplatform { namespace log { namespace detail {

inline settings to_settings(const ptree& config)
{
    const auto spdlog_cfg = detail::prepare_config(config);
    settings settings;
    settings.parse_ptree(spdlog_cfg);
    return settings;
}

inline constexpr spdlog::level::level_enum to_spdlog_level(const severity_level value)
{
    return spdlog::level::level_enum(int(spdlog::level::emerg) - int(value));
}

class service
{
public:
    service()
    {
        // TODO construct default?
    }

    service(const log::settings& settings)
    {
        init_spdlog(settings);
    }

    service(const ptree& config)
    {
        init_spdlog(to_settings(config));
    }

    service(
        const std::vector<std::shared_ptr<spdlog::logger>>& loggers,
        const std::vector<spdlog::sink_ptr>& sinks)
    {
        for (const auto& logger : loggers)
        {
            this->add(logger);
        }
        for (const auto& sink : sinks)
        {
            this->add(sink);
        }
    }

    using spdlog_logger_impl = std::shared_ptr<spdlog::logger>;

    struct typed_logger_impl
    {
        spdlog_logger_impl logger;
    };

    using typed_stream = typed::stream<spdlog_stream, char>;

    void construct(spdlog_logger_impl& impl) const
    {
        const auto global = spd_loggers_.find("global");
        if (global != spd_loggers_.end())
        {
            impl = global->second;
            return;
        }
        construct_default(impl);
    }

    void construct(spdlog_logger_impl& impl, const std::string& name) const
    {
        const auto it = spd_loggers_.find(name);
        if (it == spd_loggers_.end())
        {
            construct(impl);
        }
        else
        {
            impl = it->second;
        }
    }

    void destruct(spdlog_logger_impl&) const
    {
    }

    spdlog_stream write(const spdlog_logger_impl& logger, const severity_level level) const
    {
        return spdlog_stream(
            logger.get(),
            to_spdlog_level(level),
            logger && logger->should_log(to_spdlog_level(level)));
    }

    void construct(typed_logger_impl& impl, const std::string& name) const
    {
        construct(impl.logger, name);
    }

    void destruct(typed_logger_impl&) const
    {
    }

    typed_stream write(const typed_logger_impl& impl, const severity_level level) const
    {
        spdlog_stream spds = write(impl.logger, level);
        return typed_stream(std::move(spds));
    }

    void reopen_all() const
    {
        for (const auto& sink : spd_sinks_)
        {
            if (const auto reopenable = std::dynamic_pointer_cast<log::reopenable>(sink))
            {
                reopenable->reopen();
            }
        }
    }

    bool should_log(const spdlog_logger_impl& logger, const severity_level level) const
    {
        return logger && logger->should_log(to_spdlog_level(level));
    }

    bool should_log(const typed_logger_impl& impl, const severity_level level) const
    {
        return should_log(impl.logger, level);
    }

    void update_log_levels_from(const ptree& config)
    {
        update_spdlog_levels_from(to_settings(config));
    }

    bool has_logger(const std::string& name) const
    {
        return spd_loggers_.contains(name);
    }

private:
    std::unordered_map<std::string, spdlog_logger_impl> spd_loggers_;
    std::vector<spdlog::sink_ptr> spd_sinks_;

    void init_spdlog(const log::settings& settings)
    {
        for (auto& box : log::detail::spd::make(settings))
        {
            this->add(std::move(box.logger));
            for (auto& sink : box.sinks)
            {
                this->add(std::move(sink));
            }
        }
        auto warmup = []() { /* Some init code in the worker thread*/ };
        auto flush_interval_ms = std::chrono::milliseconds::zero();
        auto teardown = []() {};
        spdlog::set_async_mode(
            100000,
            spdlog::async_overflow_policy::block_retry,
            warmup,
            flush_interval_ms,
            teardown);
    }

    void update_spdlog_levels_from(const log::settings& settings)
    {
        for (auto&& [name, log_config] : settings.logs)
        {
            auto it = spd_loggers_.find(name);
            if (it != spd_loggers_.cend())
            {
                assert(it->second); // TODO: Add preconditions and asserts that would ensure this.
                it->second->set_level(spd::level_from_str(log_config.level));
            }
        }
    }

    void add(spdlog_logger_impl impl)
    {
        spd_loggers_.insert({ impl->name(), std::move(impl) });
    }

    void add(spdlog::sink_ptr value)
    {
        spd_sinks_.push_back(std::move(value));
    }

    void construct_default(spdlog_logger_impl& impl) const
    {
        using sink_type = spdlog::sinks::null_sink_st;
        static auto sink = std::make_shared<sink_type>();
        impl = detail::spd::make_sync_logger(
            "default", std::initializer_list<std::shared_ptr<sink_type>>({ sink }));
    }
};

class global_service
{
public:
    global_service() : impl_(std::make_shared<service>())
    {
    }

    static global_service& instance()
    {
        static global_service result;
        return result;
    }

    void init(const std::shared_ptr<log::detail::service>& value)
    {
        impl_ = value;
    }

    void deinit()
    {
        impl_.reset();
    }

    service& get() const
    {
        return *instance().impl_;
    }

    std::shared_ptr<service> get_ptr() const
    {
        return instance().impl_;
    }

    bool inited() const
    {
        return bool(instance().impl_);
    }

private:
    std::shared_ptr<service> impl_;
};

#define YGLOBAL_LOG_SERVICE ::yplatform::log::detail::global_service::instance().get()

} // namespace detail
} // namespace log
} // namespace yplatform
