#pragma once

#include <yplatform/log/service.h>
#include <yplatform/log/has_log_prefix.h>
#include <spdlog/logger.h>
#include <atomic>

namespace yplatform { namespace log {

class spdlog_logger : public has_log_prefix
{
public:
    using stream = spdlog_stream;

    spdlog_logger() : service_(detail::global_service::instance().get())
    {
        service_.construct(impl_);
    }

    spdlog_logger(log::service& service) : service_(service.get())
    {
        service_.construct(impl_);
    }

    spdlog_logger(boost::asio::io_service& io) : service_(get_service_impl(io))
    {
        service_.construct(impl_);
    }

    spdlog_logger(boost::asio::io_service& io, const std::string& name)
        : service_(get_service_impl(io))
    {
        service_.construct(impl_, name);
    }

    spdlog_logger(log::service& service, const std::string& name) : service_(service.get())
    {
        service_.construct(impl_, name);
    }

    spdlog_logger(log::service& service, const std::string& name, const std::string& prefix)
        : has_log_prefix(prefix), service_(service.get())
    {
        service_.construct(impl_, name);
    }

    spdlog_logger(log::detail::service& service) : service_(service)
    {
        service_.construct(impl_);
    }

    spdlog_logger(log::detail::service& service, const std::string& name) : service_(service)
    {
        service_.construct(impl_, name);
    }

    spdlog_logger(log::detail::service& service, const std::string& name, const std::string& prefix)
        : has_log_prefix(prefix), service_(service)
    {
        service_.construct(impl_, name);
    }

    spdlog_logger(const spdlog_logger& other)
        : has_log_prefix(other), service_(other.service_), impl_(other.impl_)
    {
    }

    spdlog_logger(spdlog_logger&& other)
        : has_log_prefix(std::move(other)), service_(other.service_), impl_(std::move(other.impl_))
    {
    }

    ~spdlog_logger()
    {
        service_.destruct(impl_);
    }

    spdlog_logger& operator=(const spdlog_logger& other)
    {
        if (this != &other)
        {
            has_log_prefix::operator=(other);
            impl_ = other.impl_;
        }
        return *this;
    }

    spdlog_logger& operator=(spdlog_logger&& other)
    {
        if (this != &other)
        {
            has_log_prefix::operator=(std::move(other));
            impl_ = std::move(other.impl_);
        }
        return *this;
    }

    stream write(const severity_level level) const
    {
        stream s(service_.write(impl_, level));
        if (s) s << get_log_prefix();
        return s;
    }

    bool should_log(const severity_level level) const
    {
        return service_.should_log(impl_, level);
    }

    template <class... Args>
    void add_attribute(Args&&...)
    {
    }

private:
    log::detail::service& service_;
    log::detail::service::spdlog_logger_impl impl_;
};

class global_spdlog_logger
{
public:
    static global_spdlog_logger& instance()
    {
        static global_spdlog_logger result;
        return result;
    }

    global_spdlog_logger() : initialized_(false), impl_(detail::global_service::instance().get())
    {
    }

    void init(log::detail::service& service)
    {
        impl_ = spdlog_logger(service);
    }

    void deinit()
    {
        // impl_.reset();
    }

    spdlog_logger& get()
    {
        return instance().impl_;
    }

private:
    std::atomic_flag initialized_;
    spdlog_logger impl_;
};

} // namespace log
} // namespace yplatform
