#pragma once

#include <yplatform/log/detail/service.h>
#include <yplatform/log/typed_stream.h>
#include <yplatform/util/update_iterator.h>
#include <boost/mpl/vector.hpp>
#include <boost/range/adaptor/map.hpp>
#include <boost/range/algorithm/copy.hpp>
#include <boost/type_erasure/builtin.hpp>
#include <boost/type_erasure/operators.hpp>
#include <boost/type_erasure/any_cast.hpp>
#include <boost/variant.hpp>
#include <algorithm> // std:move
#include <iostream>
#include <map>
#include <set>
#include <string>
#include <type_traits>
#include <utility>

#include <spdlog/details/format.h>

namespace yplatform { namespace log { namespace typed {

namespace detail {

template <
    typename CharT,
    typename Traits = std::char_traits<CharT>,
    typename Alloc = std::allocator<CharT>,
    typename Ret = std::basic_string<CharT, Traits, Alloc>,
    typename T>
inline Ret to_string(T&& value)
{
    if constexpr (std::is_convertible_v<T&&, Ret>)
    {
        return std::forward<T>(value);
    }
    else if constexpr (std::is_same_v<Ret, std::string>)
    {
        static constexpr auto fmt = std::is_same_v<std::remove_cvref_t<T>, bool> ? "{:d}" : "{}";
        return fmt::format(fmt, value);
    }
    else if constexpr (std::is_same_v<Ret, std::wstring>)
    {
        static constexpr auto fmt = std::is_same_v<std::remove_cvref_t<T>, bool> ? L"{:d}" : L"{}";
        return fmt::format(fmt, value);
    }
    else
    {
        std::basic_stringstream<CharT, Traits, Alloc> ss;
        ss << value;
        return ss.str();
    }
}

} // namespace detail

// Typed log attribute - a pair name+value.
template <
    typename CharT,
    typename Traits = std::char_traits<CharT>,
    typename Alloc = std::allocator<CharT>>
struct attr
{
    typedef std::basic_ostream<CharT, Traits> ostream_type;
    typedef std::basic_string<CharT, Traits, Alloc> string_type;

    typedef string_type name;
    typedef string_type value;

    typedef std::pair<name const, value> type;
};

// TODO: Alloc?
template <typename CharT, typename Traits>
inline std::basic_ostream<CharT, Traits>& operator<<(
    std::basic_ostream<CharT, Traits>& os,
    typename attr<CharT, Traits>::type const& ft)
{
    return os << '[' << ft.first << "=>" << ft.second << ']';
}

/// Creates attr from string and value.
/**
 * @param key attr name.
 * @param value attr value.
 * @returns attr definition instance.
 */
template <typename C, typename Tr, typename A, typename T>
typename attr<C, Tr, A>::type make_attr(std::basic_string<C, Tr, A>&& key, T&& value)
{
    return typename attr<C, Tr, A>::type(
        std::move(key), detail::to_string<C, Tr, A>(std::forward<T>(value)));
}

/// Creates attr from string and value.
/**
 * @param key attr name.
 * @param value attr value.
 * @returns attr definition instance.
 */
template <typename C, typename Tr, typename A, typename T>
typename attr<C, Tr, A>::type make_attr(std::basic_string<C, Tr, A> const& key, T&& value)
{
    return typename attr<C, Tr, A>::type(key, detail::to_string<C, Tr, A>(std::forward<T>(value)));
}

/// Creates attr from string and value.
/**
 * @param key attr name.
 * @param value attr value.
 * @returns attr definition instance.
 */
template <typename C, typename T>
typename attr<C>::type make_attr(C const* key, T&& value)
{
    return make_attr(std::basic_string<C>(key), detail::to_string<C>(std::forward<T>(value)));
}

/// Typed log attributes map.
template <typename C, typename Tr = std::char_traits<C>, typename A = std::allocator<C>>
class basic_attributes_map;

// forward declaration of friend function.
// it will be defined after class basic_attributes_map below.
template <typename C, typename Tr, typename A>
basic_attributes_map<C, Tr, A> scoped(basic_attributes_map<C, Tr, A>& map);

/// Typed log attributes map.
template <typename C, typename Tr, typename A>
class basic_attributes_map
{
public:
    typedef typename attr<C, Tr, A>::type attr_type;
    typedef typename attr<C, Tr, A>::name attr_name;
    typedef typename attr<C, Tr, A>::value attr_value;

protected:
    typedef std::map<attr_name const, attr_value> map_type;

    struct proxy;
    typedef std::shared_ptr<proxy> proxy_ptr;

    struct proxy
    {
        proxy(proxy_ptr parent = proxy_ptr()) : parent(parent)
        {
        }

        proxy_ptr parent;
        map_type map;
    };

    map_type& map()
    {
        return proxy_->map;
    }
    map_type const& map() const
    {
        return proxy_->map;
    }

    proxy_ptr& parent()
    {
        return proxy_->parent;
    }
    proxy_ptr const& parent() const
    {
        return proxy_->parent;
    }

public:
    /// Constructs attributes map based on given map.
    /**
     * @param parent parent basic_attributes_map.
     */
    basic_attributes_map(proxy_ptr parent = proxy_ptr()) : proxy_(new proxy)
    {
        proxy_->parent = parent;
    }

    /// Constructs attributes map from initiaizer_list.
    /**
     * @param attrs fileds definition list.
     * @param parent parent basic_attributes_map.
     */
    basic_attributes_map(std::initializer_list<attr_type> attrs, proxy_ptr parent = proxy_ptr())
        : proxy_(new proxy)
    {
        proxy_->parent = parent;
        proxy_->map.insert(attrs);
    }

    /// Replaces of adds given attr.
    /**
     * @param attr the attrs to replace or add.
     * @returns reference to this basic_attributes_map.
     */
    inline basic_attributes_map& replace(attr_type&& attr)
    {
        proxy_->map[std::move(attr.first)] = std::move(attr.second);
        return *this;
    }

    /// Replaces of adds given attr.
    /**
     * @param attr the attrs to replace or add.
     * @returns reference to this basic_attributes_map.
     */
    inline basic_attributes_map& replace(attr_type const& attr)
    {
        proxy_->map[attr.first] = attr.second;
        return *this;
    }

    /// Replaces of adds given attrs.
    /**
     * @param attrs list of attrs to replace or add.
     * @returns reference to this basic_attributes_map.
     */
    basic_attributes_map& replace(std::initializer_list<attr_type> attrs)
    {
        for (auto const& attr : attrs)
        {
            replace(attr);
        }

        return *this;
    }

    /// Erase given attr from internal map.
    /**
     * @param names attr name to be erased.
     * @returns reference to this basic_attributes_map.
     */
    inline basic_attributes_map& erase(attr_name const& name)
    {
        proxy_->map.erase(name);
        return *this;
    }

    /// Removes given attrs from internal map.
    /**
     * @param names list of the names.
     * @returns reference to this basic_attributes_map.
     */
    basic_attributes_map& erase(std::initializer_list<attr_name> names)
    {
        for (auto const& name : names)
            erase(name);

        return *this;
    }

protected:
    // template <typename C, typename Tr, typename A>
    friend inline boost::optional<attr_value const&> cascade_find(
        basic_attributes_map const& map,
        attr_name const& name)
    {
        return map.cascade_find(name);
    }

    friend inline std::set<attr_name> cascade_keys(basic_attributes_map const& map)
    {
        return map.cascade_keys();
    }

    boost::optional<attr_value const&> cascade_find(attr_name const& name) const
    {
        for (proxy_ptr proxy = proxy_; proxy; proxy = proxy->parent)
        {
            typename map_type::const_iterator found = proxy->map.find(name);
            if (found != proxy->map.end()) return found->second;
        }

        return boost::optional<attr_value const&>();
    }

    std::set<attr_name> cascade_keys() const
    {
        std::set<attr_name> keys;

        for (proxy_ptr proxy = proxy_; proxy; proxy = proxy->parent)
        {
            boost::copy(proxy->map | boost::adaptors::map_keys, std::inserter(keys, keys.end()));
        }

        return keys;
    }

private:
    friend basic_attributes_map scoped<>(basic_attributes_map& map);

    proxy_ptr proxy_;
};

typedef basic_attributes_map<char> attributes_map;
typedef basic_attributes_map<wchar_t> wattributes_map;

/// Create new scope from given map.
/**
 * @param map attributes map.
 * @returns new scope based on map.
 */
template <typename C, typename Tr, typename A>
inline basic_attributes_map<C, Tr, A> scoped(basic_attributes_map<C, Tr, A>& map)
{
    return basic_attributes_map<C, Tr, A>(map.proxy_);
}

/// Add or replace attr in given map.
/**
 * @param map attributes map.
 * @param attr attr to add or replace.
 * @returns Reference to map.
 */
template <typename C, typename Tr, typename A>
inline basic_attributes_map<C, Tr, A>& operator<<(
    basic_attributes_map<C, Tr, A>& map,
    typename attr<C, Tr, A>::type const& attr)
{
    return map.replace(attr);
}

} // namespace typed
} // namespace log
} // namespace yplatform
