#pragma once

#include <yplatform/log/typed_attributes.h>
#include <algorithm>
#include <map>
#include <string>

namespace yplatform { namespace log { namespace typed {

/// Typed stream wraps any another logger's stream and
/// just dumps TSKV pairs to this stream.
template <
    typename Impl,
    typename C,
    typename Tr = std::char_traits<C>,
    typename A = std::allocator<C>>
class stream : public boost::noncopyable
{
    typedef basic_attributes_map<C, Tr, A> attributes_map_type;

public:
    stream(Impl&& impl) : impl(std::forward<Impl>(impl)), first_elem(true)
    {
    }

    stream(stream&& other) : impl(std::move(other.impl)), first_elem(other.first_elem)
    {
    }

    stream& operator<<(const attributes_map_type& amap)
    {
        for (auto const& aname : cascade_keys(amap))
        {
            if (auto optional_aval = cascade_find(amap, aname))
                if (!aname.empty())
                {
                    if (!first_elem) impl << '\t';
                    else
                        first_elem = false;
                    escape_and_write(aname, impl);
                    impl << '=';
                    escape_and_write(*optional_aval, impl);
                }
        }
        return *this;
    }

    stream& operator<<(const typename attr<C, Tr, A>::type& attr)
    {
        if (!attr.first.empty())
        {
            if (!first_elem) impl << '\t';
            else
                first_elem = false;
            escape_and_write(attr.first, impl);
            impl << '=';
            escape_and_write(attr.second, impl);
        }
        return *this;
    }

    operator bool() const
    {
        return bool(impl);
    }

    void close()
    {
        impl.close();
    }

private:
    void escape_and_write(std::basic_string<C, Tr, A> s, Impl& impl)
    {
        s.erase(
            std::remove_if(
                s.begin(),
                s.end(),
                [](C c) { return Tr::eq(c, '\t') || Tr::eq(c, '\r') || Tr::eq(c, '\n'); }),
            s.end());
        impl << s;
    }

    Impl impl;
    bool first_elem;
};

} // namespace typed
} // namespace log
} // namespace yplatform
