#pragma once

#include <yplatform/log.h>
#include <yplatform/net/socket.h>

namespace yplatform { namespace net {

template <typename Protocol, typename Settings>
class [[deprecated]] acceptor
    : public Protocol::acceptor
    , public log::contains_logger
{
    typedef Protocol protocol_t;
    typedef typename protocol_t::acceptor acceptor_t;
    typedef Settings settings_t;

public:
    acceptor(boost::asio::io_service * io, const settings_t& settings)
        : acceptor_t(*io), settings_(settings)
    {
    }

    inline const settings_t& settings() const
    {
        return settings_;
    }
    inline settings_t& settings()
    {
        return settings_;
    }

private:
    settings_t settings_;
};

template <typename Socket>
class basic_acceptor
{
public:
    using socket_type = Socket;
    using endpoint_type = typename Socket::endpoint_t;
    using protocol_type = typename Socket::raw_socket_t::protocol_type;
    using implementation_type = typename protocol_type::acceptor;

    basic_acceptor(boost::asio::io_service& io, const endpoint_type& endpoint) : impl(io, endpoint)
    {
    }

    basic_acceptor(boost::asio::io_service& io) : impl(io)
    {
    }

    template <typename Dummy = void>
    typename std::enable_if<std::is_same<socket_type, tcp_socket>::value, Dummy>::type bind(
        const endpoint_type& endpoint,
        const acceptor_settings& settings)
    {
        boost::system::error_code ec;
        impl.open(endpoint.protocol(), ec);
        if (ec) throw std::runtime_error("acceptor open failed with " + ec.message());

        if (endpoint.protocol() == boost::asio::ip::tcp::v6() && settings.ipv6_only)
            impl.set_option(boost::asio::ip::v6_only(true));

        impl.set_option(typename implementation_type::reuse_address(true));

        if (settings.reuse_port)
        {
#ifdef SO_REUSEPORT // this flag is not supported in libc for kernels < 3.9 (e.g. in arcadia)
            const int reuse = 1;
            if (::setsockopt(
                    impl.native_handle(), SOL_SOCKET, SO_REUSEPORT, &reuse, sizeof(reuse)) < 0)
            {
                std::ostringstream stream;
                stream << "setsockopt SOL_SOCKET SO_REUSEPORT " << reuse
                       << " failed: " << strerror(errno);
                throw std::runtime_error(stream.str());
            }
#else
            throw std::runtime_error("SO_REUSEPORT is not supported");
#endif
        }

#ifdef SOL_TCP
        if (settings.defer_accept)
        {
            const auto timeout = settings.read_timeout.count();
            ::setsockopt(
                impl.native_handle(), SOL_TCP, TCP_DEFER_ACCEPT, &timeout, sizeof(timeout));
        }
#endif
        impl.bind(endpoint);
        impl.listen(settings.listen_backlog);
    }

    void accept(Socket& socket)
    {
        impl.accept(socket.raw_socket());
    }

    template <typename Handler>
    void async_accept(Socket& socket, Handler&& handler)
    {
        impl.async_accept(socket.raw_socket(), std::forward<Handler>(handler));
    }

    template <typename Handler>
    void async_accept(Socket& socket, endpoint_type& ep, Handler&& handler)
    {
        impl.async_accept(socket.raw_socket(), ep, std::forward<Handler>(handler));
    }

    bool is_open() const
    {
        return impl.is_open();
    }

    void close()
    {
        boost::system::error_code ec;
        impl.close(ec);
    }

    void close(boost::system::error_code& ec)
    {
        return impl.close(ec);
    }

    implementation_type& raw_acceptor()
    {
        return impl;
    }

private:
    implementation_type impl;
};

using tcp_acceptor = basic_acceptor<tcp_socket>;
using unix_socket_acceptor = basic_acceptor<unix_socket>;

}}
