#pragma once

#include <yplatform/net/buffers/chunk.h>
#include <boost/asio.hpp>
#include <boost/make_shared.hpp>
#include <deque>

namespace yplatform { namespace net {

template <typename T, typename Allocator = std::allocator<void>>
class zcopy_list
{
public:
    typedef T chunk_t;
    typedef typename std::allocator_traits<Allocator>::template rebind_alloc<chunk_t> allocator_type;
    typedef std::deque<chunk_t, allocator_type> cont_t;
    typedef typename cont_t::iterator iterator;
    typedef typename cont_t::const_iterator const_iterator;
    typedef typename cont_t::pointer pointer;
    typedef typename cont_t::const_pointer const_pointer;
    typedef typename cont_t::reference reference;
    typedef typename cont_t::const_reference const_reference;

    zcopy_list() : impl_(boost::make_shared<cont_t>()), size_(0)
    {
    }

    zcopy_list(allocator_type const& alloc) : impl_(boost::allocate_shared<cont_t>(alloc)), size_(0)
    {
    }

    iterator begin()
    {
        return impl_->begin();
    }
    const_iterator begin() const
    {
        return impl_->begin();
    }

    iterator end()
    {
        return impl_->end();
    }
    const_iterator end() const
    {
        return impl_->end();
    }

    void clear()
    {
        impl_->clear();
        size_ = 0;
    }

    std::size_t size() const
    {
        return size_;
    }

    bool empty() const
    {
        return size_ == 0;
    }

    void push_back(const T& v)
    {
        impl_->push_back(v);
        size_ += boost::asio::buffer_size(v);
    }
    void push_front(const T& v)
    {
        impl_->push_front(v);
        size_ += boost::asio::buffer_size(v);
    }

    void pop_back()
    {
        size_ -= boost::asio::buffer_size(impl_->back());
        impl_->pop_back();
    }

    void pop_front()
    {
        size_ -= boost::asio::buffer_size(impl_->front());
        impl_->pop_front();
    }

    reference front()
    {
        return impl_->front();
    }
    const_reference front() const
    {
        return impl_->front();
    }

    reference back()
    {
        return impl_->back();
    }
    const_reference back() const
    {
        return impl_->back();
    }

    void consume(std::size_t size)
    {
        while (impl_->size() && size >= boost::asio::buffer_size(impl_->front()))
        {
            size -= boost::asio::buffer_size(impl_->front());
            pop_front();
        }
        if (size == 0 || impl_->empty()) return;
        size_ -= size;
        impl_->front().consume(size);
    }

    void append_and_clear(cont_t& c)
    {
        for (typename cont_t::const_iterator i = c.begin(), i_end = c.end(); i != i_end; ++i)
            size_ += boost::asio::buffer_size(*i);
        if (impl_->empty()) impl_->swap(c);
        else
        {
            impl_->insert(impl_->end(), c.begin(), c.end());
            c.clear();
        }
    }

private:
    boost::shared_ptr<cont_t> impl_;
    std::size_t size_;
};

template <typename Buffer, typename Allocator = std::allocator<void>>
class output_buffer_sequence
{
public:
    typedef Buffer buffer_t;
    typedef typename std::allocator_traits<Allocator>::template rebind_alloc<buffer_t> allocator_type;
    typedef zcopy_list<buffer_t, Allocator> send_queue_t;
    typedef typename zcopy_list<buffer_t, Allocator>::cont_t wait_queue_t;

    output_buffer_sequence(allocator_type const& alloc = allocator_type())
        : send_queue_(alloc), wait_queue_(alloc)
    {
    }

    ~output_buffer_sequence()
    {
        clear();
    }

    const send_queue_t& send_queue() const
    {
        return send_queue_;
    }

    const wait_queue_t& wait_queue() const
    {
        return wait_queue_;
    }

    bool push(const buffer_t& buff)
    {
        wait_queue_.push_back(buff);
        return send_queue_.empty();
    }

    bool push_back(const buffer_t& buff)
    {
        return push(buff);
    }

    void consume(std::size_t bytes)
    {
        send_queue_.consume(bytes);
    }

    bool flush(std::size_t max_packet_size = 0)
    {
        if (wait_queue_.empty()) return true;
        if (max_packet_size == 0)
        {
            send_queue_.append_and_clear(wait_queue_);
            return false;
        }
        for (std::size_t current_size = send_queue_.size(); current_size < max_packet_size;
             current_size += boost::asio::buffer_size(wait_queue_.front()))
        {
            if (wait_queue_.empty()) break;
            send_queue_.push_back(wait_queue_.front());
            wait_queue_.pop_front();
        }
        return send_queue_.empty();
    }

    void clear()
    {
        send_queue_.clear();
        wait_queue_.clear();
    }

private:
    send_queue_t send_queue_;
    wait_queue_t wait_queue_;
};

}}

namespace boost { namespace asio {

template <typename T, typename A>
struct is_const_buffer_sequence<yplatform::net::zcopy_list<T, A>> : std::true_type
{
};

}}
