#pragma once

#include <boost/shared_ptr.hpp>
#include <boost/asio/buffer.hpp>
#include <yplatform/string_config.h>
#include <yplatform/zerocopy/chunk.h>

namespace yplatform { namespace net { namespace buffers {

using ::yplatform::zerocopy::base_chunk;
using ::yplatform::zerocopy::const_base_chunk;
using ::yplatform::zerocopy::base_chunk_ptr;
using ::yplatform::zerocopy::const_base_chunk_ptr;

class chunk_buffer
{
public:
    typedef char byte_t;

    chunk_buffer(const base_chunk_ptr& owner) : data_(owner->buff()), owner_(owner)
    {
    }

    void* data(void) const
    {
        return data_.first;
    }
    std::size_t size(void) const
    {
        return data_.second;
    }

private:
    std::pair<byte_t*, std::size_t> data_;
    base_chunk_ptr owner_;
};

class const_chunk_buffer
{
public:
    typedef char byte_t;

    const_chunk_buffer() : data_(0, 0)
    {
    }

    const_chunk_buffer(const const_base_chunk_ptr& owner) : data_(owner->buff()), owner_(owner)
    {
    }

    const void* data(void) const
    {
        return data_.first;
    }
    std::size_t size(void) const
    {
        return data_.second;
    }
    void consume(std::size_t bytes)
    {
        if (bytes < data_.second)
        {
            data_.first += bytes;
            data_.second -= bytes;
        }
        else
        {
            data_.first += data_.second;
            data_.second = 0;
        }
    }

    operator boost::asio::const_buffer() const
    {
        return boost::asio::const_buffer(data(), size());
    }

private:
    std::pair<const byte_t*, std::size_t> data_;
    const_base_chunk_ptr owner_;
};

inline chunk_buffer make_chunk_buffer(base_chunk_ptr chunk)
{
    return chunk_buffer(chunk);
}

inline chunk_buffer make_chunk_buffer(base_chunk* chunk)
{
    base_chunk_ptr tmp(chunk);
    return make_chunk_buffer(tmp);
}

inline const_chunk_buffer make_chunk_buffer(const_base_chunk_ptr chunk)
{
    return const_chunk_buffer(chunk);
}

inline const_chunk_buffer make_chunk_buffer(const_base_chunk* chunk)
{
    const_base_chunk_ptr tmp(chunk);
    return make_chunk_buffer(tmp);
}

}}}

namespace boost { namespace asio {

template <typename PointerToPodType>
inline PointerToPodType buffer_cast(const yplatform::net::buffers::const_chunk_buffer& b)
{
    return static_cast<PointerToPodType>(b.data());
}

template <typename PointerToPodType>
inline PointerToPodType buffer_cast(const yplatform::net::buffers::chunk_buffer& b)
{
    return static_cast<PointerToPodType>(b.data());
}

inline std::size_t buffer_size(const yplatform::net::buffers::const_chunk_buffer& b)
{
    return b.size();
}

inline std::size_t buffer_size(const yplatform::net::buffers::chunk_buffer& b)
{
    return b.size();
}

}}

#include <yplatform/net/buffers/chunk_impl.h>
