#pragma once

#include <vector>
#include <boost/bind.hpp>
#include <boost/make_shared.hpp>
#include <yplatform/net/buffers/chunk.h>

namespace yplatform { namespace net { namespace buffers {

template <typename StdCont>
class const_stdcont_chunk : public const_base_chunk
{
public:
    explicit const_stdcont_chunk(StdCont& str)
    {
        data_.swap(str);
    }

    virtual std::pair<const byte_t*, std::size_t> buff()
    {
        return std::make_pair(data_.data(), data_.size());
    }

private:
    StdCont data_;
};

template <typename StdCont>
inline const_chunk_buffer make_const_stdcont_chunk(StdCont& cont)
{
    boost::shared_ptr<const_base_chunk> ptr(new const_stdcont_chunk<StdCont>(cont));
    return make_chunk_buffer(ptr);
}

template <typename StdCont>
inline const_chunk_buffer make_const_cstdcont_chunk(const StdCont& cont)
{
    StdCont temp_cont(cont);
    return make_const_stdcont_chunk(temp_cont);
}

class const_asio_chunk : public const_base_chunk
{
public:
    explicit const_asio_chunk(const boost::asio::const_buffer& buff) : data_(buff)
    {
    }

    ~const_asio_chunk()
    {
        delete[] boost::asio::buffer_cast<const char*>(data_);
    }

    virtual std::pair<const byte_t*, std::size_t> buff()
    {
        return std::make_pair(
            boost::asio::buffer_cast<const byte_t*>(data_), boost::asio::buffer_size(data_));
    }

private:
    boost::asio::const_buffer data_;
};

template <class Allocator>
class allocated_const_asio_chunk : public const_base_chunk
{
    typedef typename std::allocator_traits<Allocator>::template rebind_alloc<char> allocator_type;

public:
    explicit allocated_const_asio_chunk(
        const boost::asio::const_buffer& buff,
        Allocator const& alloc)
        : data_(buff), alloc_(alloc)
    {
    }

    ~allocated_const_asio_chunk()
    {
        const char* data = boost::asio::buffer_cast<const char*>(data_);
        alloc_.deallocate(const_cast<char*>(data), boost::asio::buffer_size(data_));
    }

    virtual std::pair<const byte_t*, std::size_t> buff()
    {
        return std::make_pair(
            boost::asio::buffer_cast<const byte_t*>(data_), boost::asio::buffer_size(data_));
    }

private:
    boost::asio::const_buffer data_;
    allocator_type alloc_;
};

inline const_chunk_buffer make_const_asio_chunk(const boost::asio::const_buffer& cont)
{
    boost::shared_ptr<const_base_chunk> ptr(new const_asio_chunk(cont));
    return make_chunk_buffer(ptr);
}

template <typename Allocator>
inline const_chunk_buffer allocate_const_asio_chunk(
    const Allocator& alloc,
    const boost::asio::const_buffer& cont)
{
    const_base_chunk_ptr ptr =
        boost::allocate_shared<allocated_const_asio_chunk<Allocator>>(alloc, cont, alloc);
    return make_chunk_buffer(ptr);
}

class const_raw_ptr_chunk : public const_base_chunk
{
public:
    explicit const_raw_ptr_chunk(const void* buff, std::size_t s) : data_(buff, s)
    {
    }

    virtual std::pair<const byte_t*, std::size_t> buff()
    {
        return std::make_pair(
            boost::asio::buffer_cast<const byte_t*>(data_), boost::asio::buffer_size(data_));
    }

private:
    boost::asio::const_buffer data_;
};

template <typename Iter, typename FragIter, typename Func>
void for_each_chunk(Iter begin, Iter end, FragIter head, FragIter tail, Func f)
{
    typedef yplatform::zerocopy::const_base_chunk::byte_t byte_t;
    if (begin == end) return;
    Iter i = begin;
    Iter next = begin;
    for (++next; i != end; ++i)
    {
        if (i == begin || next == end)
        {
            const byte_t* data_start = (i == begin ? head : (*i)->buff().first);
            const byte_t* data_end =
                (next == end ? tail : (*i)->buff().first + (*i)->buff().second);
            boost::shared_ptr<const_base_chunk> ptr(
                new const_raw_ptr_chunk(data_start, data_end - data_start));
            f(ptr);
        }
        else
        {
            f(*i);
        }
        if (next != end) ++next;
    }
}

template <typename Iter, typename FragIter, typename Queue>
void insert_chunks_2_queue(Iter begin, Iter end, FragIter head, FragIter tail, Queue& queue)
{
    const_chunk_buffer (*func)(const_base_chunk_ptr) = &make_chunk_buffer;
    for_each_chunk(
        begin, end, head, tail, boost::bind(&Queue::push_back, &queue, boost::bind(func, _1)));
}

}}}
