#pragma once

#include <yplatform/net/session_strand.h>
#include <yplatform/net/settings.h>
#include <yplatform/net/handlers/mem_alloc.h>
#include <yplatform/net/handlers/timer_handler.h>
#include <yplatform/net/dns/resolver.h>

namespace yplatform { namespace net {

enum address_family
{
    ipv4,
    ipv6
};

template <typename Settings = client_settings, typename Protocol = boost::asio::ip::tcp>
class client_session_strand : public session_strand<Settings, Protocol>
{
public:
    typedef session_strand<Settings, Protocol> base_t;
    typedef typename base_t::protocol_t protocol_t;
    typedef typename base_t::settings_t settings_t;

    client_session_strand(io_data* io_data, const settings_t& settings)
        : base_t(io_data, settings, boost::asio::ssl::stream_base::client)
        , resolver_(*io_data->get_io())
        , connect_timer_(*io_data->get_io())
    {
    }

    typedef dns::resolver resolver_t;
    typedef resolver_t::iterator_a iterator_ipv4;
    typedef resolver_t::iterator_aaaa iterator_ipv6;

    //
    // Calls logic example for settings_t::resolve_order=="ipv4_ipv6".
    //
    //                      resolve ipv4
    //                           |
    //            no addresses -- -- got addresses
    //              or error                |
    //                |          iterate and try to connect (ignore errors)
    //                |                     |
    //                |           no more   |
    //                |          addresses - - success
    //                |                |         |
    //                |                |         |
    //                |<---------------         exit
    //                |
    //            resolve ipv6
    //                |
    //   no addresses- - got addresses ----> iterate and try to connect
    //     or error                                |
    //       |                                    exit
    //      exit
    template <typename Handler, typename DeadlineHandler>
    void connect(
        const string& hostname,
        short unsigned port,
        Handler&& handler,
        DeadlineHandler&& deadline_handler);

    void cancel_connect();

    struct timings
    {
        typedef time_traits::time_point time_point;
        typedef time_traits::duration duration;

        timings()
            : start_(undefined_time()), resolved_(undefined_time()), connected_(undefined_time())
        {
        }

        void set_start()
        {
            start_ = now();
        }
        void set_resolved()
        {
            resolved_ += now() - start_;
        }
        void set_connected()
        {
            connected_ += now() - start_;
        }

        duration get_resolve_time() const
        {
            return resolved_;
        }
        duration get_connect_time() const
        {
            return connected_;
        }
        duration undefined_time() const
        {
            return duration::zero();
        }

    private:
        time_point start_;
        duration resolved_;
        duration connected_;

        time_point now()
        {
            return time_traits::clock::now();
        }
    };

    const timings& get_timings() const;

    resolver_t& get_resolver()
    {
        return resolver_;
    }

private:
    template <typename Handler, typename DeadlineHandler>
    void connect_protected(
        const string& hostname,
        short unsigned port,
        Handler&& handler,
        DeadlineHandler&& deadline_handler);

    template <address_family af, typename Handler, typename DeadlineHandler>
    void do_one_step_connect(
        const string& hostname,
        short unsigned port,
        Handler&& handler,
        DeadlineHandler&& deadline_handler);

    template <address_family af1, address_family af2, typename Handler, typename DeadlineHandler>
    void do_two_step_connect(
        const string& hostname,
        short unsigned port,
        Handler&& handler,
        DeadlineHandler&& deadline_handler);

    template <typename Iterator, typename Handler, typename DeadlineHandler>
    void handle_resolve(
        const error_code& e,
        Iterator i,
        const Handler& handler,
        const DeadlineHandler& deadline_handler,
        short unsigned port,
        const string& hostname);

    template <typename Handler, typename DeadlineHandler>
    void do_socket_async_connect(
        const typename protocol_t::endpoint& endpoint,
        Handler&& handler,
        DeadlineHandler&& deadline_handler);

    template <typename Iterator, typename Handler, typename DeadlineHandler>
    void handle_connect(
        const error_code& e,
        Iterator i,
        const Handler& handler,
        const DeadlineHandler& deadline_handler,
        short unsigned port);

    template <typename Handler>
    void handle_fast_connect(const error_code& e, const Handler& handler, short unsigned port);

    template <typename Iterator>
    bool empty(const Iterator& iterator);

    resolver_t resolver_;
    time_traits::timer connect_timer_;
    timings timings_;
};

namespace detail {
template <address_family, typename Resolver>
struct resolve_step_helper;

template <typename Resolver>
struct resolve_step_helper<ipv4, Resolver>
{
    template <typename Handler>
    static void resolve(Resolver& resolver, const std::string& hostname, Handler&& handler)
    {
        resolver.async_resolve_a(hostname, std::forward<Handler>(handler));
    }
    typedef typename Resolver::iterator_a iterator;
};

template <typename Resolver>
struct resolve_step_helper<ipv6, Resolver>
{
    template <typename Handler>
    static void resolve(Resolver& resolver, const std::string& hostname, Handler&& handler)
    {
        resolver.async_resolve_aaaa(hostname, std::forward<Handler>(handler));
    }
    typedef typename Resolver::iterator_aaaa iterator;
};
}

template <typename S, typename P>
template <typename Handler, typename DeadlineHandler>
void client_session_strand<S, P>::connect(
    const string& hostname,
    short unsigned port,
    Handler&& handler,
    DeadlineHandler&& deadline_handler)
{
    connect_protected(
        hostname,
        port,
        protect_handler(std::forward<Handler>(handler)),
        protect_handler(std::forward<DeadlineHandler>(deadline_handler)));
}

template <typename S, typename P>
void client_session_strand<S, P>::cancel_connect()
{
    assert(this->strand().running_in_this_thread());
    resolver_.cancel();
    base_t::cancel_operations();
}

template <typename S, typename P>
template <typename Handler, typename DeadlineHandler>
void client_session_strand<S, P>::connect_protected(
    const string& hostname,
    short unsigned port,
    Handler&& handler,
    DeadlineHandler&& deadline_handler)
{
    timings_.set_start();
    error_code ec;
    boost::asio::ip::address ip = boost::asio::ip::address::from_string(hostname, ec);
    if (!ec)
    {
        typename protocol_t::endpoint point(ip, port);
        do_socket_async_connect(
            point,
            boost::bind(
                &client_session_strand::handle_fast_connect<Handler>,
                this,
                _1,
                std::forward<Handler>(handler),
                port),
            std::forward<DeadlineHandler>(deadline_handler));
    }
    else
    {
        switch (this->settings().resolve_order)
        {
        case client_settings::ipv4:
            do_one_step_connect<ipv4>(
                hostname,
                port,
                std::forward<Handler>(handler),
                std::forward<DeadlineHandler>(deadline_handler));
            break;
        case client_settings::ipv6:
            do_one_step_connect<ipv6>(
                hostname,
                port,
                std::forward<Handler>(handler),
                std::forward<DeadlineHandler>(deadline_handler));
            break;
        case client_settings::ipv4_ipv6:
            do_two_step_connect<ipv4, ipv6>(
                hostname,
                port,
                std::forward<Handler>(handler),
                std::forward<DeadlineHandler>(deadline_handler));
            break;
        default: // ipv6_ipv4 or enum has been modified
            do_two_step_connect<ipv6, ipv4>(
                hostname,
                port,
                std::forward<Handler>(handler),
                std::forward<DeadlineHandler>(deadline_handler));
        };
    }
}

template <typename S, typename P>
template <address_family af, typename Handler, typename DeadlineHandler>
void client_session_strand<S, P>::do_one_step_connect(
    const string& hostname,
    short unsigned port,
    Handler&& handler,
    DeadlineHandler&& deadline_handler)
{
    typedef detail::resolve_step_helper<af, resolver_t> step_t;
    timings_.set_start();
    connect_timer_.expires_from_now(this->settings().resolve_timeout);
    connect_timer_.async_wait(this->strand().wrap(create_timer_handler(deadline_handler)));
    step_t::resolve(
        resolver_,
        hostname,
        this->strand().wrap(boost::bind(
            &client_session_strand::
                handle_resolve<typename step_t::iterator, Handler, DeadlineHandler>,
            this,
            _1,
            _2,
            handler,
            deadline_handler,
            port,
            hostname)));
}

template <typename S, typename P>
template <address_family af1, address_family af2, typename Handler, typename DeadlineHandler>
void client_session_strand<S, P>::do_two_step_connect(
    const string& hostname,
    short unsigned port,
    Handler&& handler,
    DeadlineHandler&& deadline_handler)
{
    do_one_step_connect<af1>(
        hostname,
        port,
        [this, hostname, port, handler, deadline_handler](const error_code& e) {
            if (!e || e == boost::asio::error::operation_aborted)
            {
                handler(e);
            }
            else
            {
                this->do_one_step_connect<af2>(hostname, port, handler, deadline_handler);
            }
        },
        deadline_handler);
}

template <typename S, typename P>
template <typename Handler, typename DeadlineHandler>
void client_session_strand<S, P>::do_socket_async_connect(
    const typename protocol_t::endpoint& endpoint,
    Handler&& handler,
    DeadlineHandler&& deadline_handler)
{
    connect_timer_.expires_from_now(this->settings().connect_timeout);
    connect_timer_.async_wait(
        this->strand().wrap(create_timer_handler(std::forward<DeadlineHandler>(deadline_handler))));
    this->raw_socket().async_connect(endpoint, this->strand().wrap(std::forward<Handler>(handler)));
}

template <typename S, typename P>
const typename client_session_strand<S, P>::timings& client_session_strand<S, P>::get_timings()
    const
{
    return timings_;
}

template <typename S, typename P>
template <typename Iterator, typename Handler, typename DeadlineHandler>
void client_session_strand<S, P>::handle_resolve(
    const error_code& e,
    Iterator i,
    const Handler& handler,
    const DeadlineHandler& deadline_handler,
    short unsigned port,
    const string& hostname)
{
    timings_.set_resolved();
    connect_timer_.cancel();
    if (e)
    {
        NET_SESSION_LOG(error) << "handle_resolve: error: " << e.message()
                               << ", hostname: " << hostname;
        return handler(e);
    }
    else if (empty(i))
    {
        NET_SESSION_LOG(error) << "handle_resolve: error: empty resolve list"
                               << ", hostname: " << hostname;
        return handler(e);
    }
    else
    {
        typename protocol_t::endpoint point(boost::asio::ip::address::from_string(*i), port);
        timings_.set_start();
        do_socket_async_connect(
            point,
            boost::bind(
                &client_session_strand::handle_connect<Iterator, Handler, DeadlineHandler>,
                this,
                _1,
                i,
                handler,
                deadline_handler,
                port),
            deadline_handler);
    }
}

template <typename S, typename P>
template <typename Iterator, typename Handler, typename DeadlineHandler>
void client_session_strand<S, P>::handle_connect(
    const error_code& e,
    Iterator i,
    const Handler& handler,
    const DeadlineHandler& deadline_handler,
    short unsigned port)
{
    timings_.set_connected();
    connect_timer_.cancel();
    if (!e)
    {
        error_code ec;
        if (this->open(ec))
        {
            NET_SESSION_LOG(error)
                << "handle_connect: error: " << ec.message() << ", ip=" << *i << " : " << port;
        }
        handler(ec);
        return;
    }

    if (e == boost::asio::error::operation_aborted)
    {
        handler(e);
        return;
    }

    Iterator i_saved = i++;
    if (!empty(i))
    {
        NET_SESSION_LOG(warning) << "handle_connect warning: use fallback endpoint. message: "
                                 << e.message() << ", ip=" << *i_saved << " : " << port;
        this->raw_socket().close();
        typename protocol_t::endpoint point(boost::asio::ip::address::from_string(*i), port);

        do_socket_async_connect(
            point,
            boost::bind(
                &client_session_strand::handle_connect<Iterator, Handler, DeadlineHandler>,
                this,
                _1,
                i,
                handler,
                deadline_handler,
                port),
            deadline_handler);
    }
    else
    {
        NET_SESSION_LOG(error) << "handle_connect: error: " << e.message() << ", ip=" << *i_saved
                               << " : " << port;
        this->raw_socket().close();
        handler(e);
    }
}

template <typename S, typename P>
template <typename Handler>
void client_session_strand<S, P>::handle_fast_connect(
    const error_code& e,
    const Handler& handler,
    short unsigned /*port*/)
{
    connect_timer_.cancel();
    if (!e)
    {
        error_code ec;
        if (this->open(ec)) NET_SESSION_LOG(error) << "handle_connect: error: " << ec.message();
        handler(ec);
    }
    else
    {
        NET_SESSION_LOG(error) << "handle_connect: error: " << e.message();
        handler(e);
    }
}

template <typename S, typename P>
template <typename Iterator>
bool client_session_strand<S, P>::empty(const Iterator& iterator)
{
    return iterator == Iterator();
}

}}
