#pragma once

#include <boost/asio/io_service.hpp>
#include <boost/asio/io_service_strand.hpp>
#include <boost/asio/ip/tcp.hpp>
#include <boost/iterator/iterator_facade.hpp>

namespace yplatform { namespace net { namespace dns { namespace detail {

// This implementation is based on boost asio resolver.
// Remember that asio resolver internal service creates a background thread
// and resolves addresses one-by-one in blocking manner using getaddrinfo().
class asio_resolver_service
{
    typedef boost::asio::ip::tcp::resolver asio_resolver_type;
    typedef boost::asio::ip::tcp::resolver::query query_type;
    typedef boost::asio::ip::tcp::resolver::iterator asio_iterator_type;

public:
    // Just a wrapper for asio resolver.
    struct resolver
    {
        std::shared_ptr<asio_resolver_type> asio_resolver;
        boost::asio::io_service::strand* strand = nullptr;
    };

    class iterator;
    // Use one iterator wrapper to cover all interface iterators.
    typedef iterator iterator_a;
    typedef iterator iterator_aaaa;
    typedef iterator iterator_ptr;
    typedef iterator iterator_mx;
    typedef iterator iterator_txt;

    asio_resolver_service(boost::asio::io_service& io_service) : io_service(io_service)
    {
    }

    void construct(resolver& resolver)
    {
        resolver.asio_resolver = std::make_shared<boost::asio::ip::tcp::resolver>(io_service);
    }

    void move_construct(resolver& r, resolver& other_resolver)
    {
        r.asio_resolver = std::move(other_resolver.asio_resolver);
    }

    void destroy(resolver& resolver)
    {
        // Resolver may have been moved from.
        if (resolver.asio_resolver)
        {
            resolver.asio_resolver->cancel();
            resolver.asio_resolver.reset();
        }
    }

    void shutdown_service()
    {
    }

    template <typename Handler>
    void async_resolve_a(resolver& resolver, const std::string& q, Handler&& handler);

    template <typename Handler>
    void async_resolve_aaaa(resolver& resolver, const std::string& q, Handler&& handler);

    template <typename Handler>
    void async_resolve_ptr(resolver& resolver, const std::string& q, Handler&& handler);

    template <typename Handler>
    void async_resolve_mx(resolver&, const std::string&, Handler&& handler);

    template <typename Handler>
    void async_resolve_txt(resolver&, const std::string&, Handler&& handler);

    void cancel(resolver& resolver)
    {
        resolver.asio_resolver->cancel();
    }

private:
    template <typename Query, typename Handler>
    void async_resolve(resolver& resolver, const Query& q, bool hostname, Handler&& handler);

    template <typename Handler>
    void post(resolver& resolver, Handler&& handler);

    boost::asio::io_service& io_service;
};

// Wrapper for asio resolver iterator, returns a string instead of asio endpoint.
class asio_resolver_service::iterator
    : public boost::iterator_facade<iterator, const std::string, boost::forward_traversal_tag>
{
    typedef asio_resolver_type::iterator asio_resolver_iterator;

public:
    iterator(bool hostname = true) : hostname(hostname)
    {
    }

    iterator(asio_resolver_iterator it, bool hostname = false) : hostname(hostname), impl(it)
    {
        update_value();
    }

    // What we implement below is determined by the boost::forward_traversal_tag
    // template parameter.
private:
    friend class boost::iterator_core_access; // For iterator_facade access to private functions.

    void increment()
    {
        ++impl;
        update_value();
    }

    bool equal(iterator const& other) const
    {
        return this->impl == other.impl;
    }

    const std::string& dereference() const
    {
        return value;
    }

    void update_value()
    {
        if (impl != asio_resolver_iterator())
        {
            value.assign(hostname ? impl->host_name() : impl->endpoint().address().to_string());
        }
        else if (!value.empty())
        {
            value.clear();
        }
    }

private:
    bool hostname = false;
    asio_resolver_iterator impl;
    std::string value;
};

template <typename Handler>
void asio_resolver_service::async_resolve_a(
    resolver& resolver,
    const std::string& q,
    Handler&& handler)
{
    query_type query(boost::asio::ip::tcp::v4(), q, "0", query_type::numeric_service);
    async_resolve(resolver, query, false, std::forward<Handler>(handler));
}

template <typename Handler>
void asio_resolver_service::async_resolve_aaaa(
    resolver& resolver,
    const std::string& q,
    Handler&& handler)
{
    query_type query(boost::asio::ip::tcp::v6(), q, "0", query_type::numeric_service);
    async_resolve(resolver, query, false, std::forward<Handler>(handler));
}

template <typename Handler>
void asio_resolver_service::async_resolve_ptr(
    resolver& resolver,
    const std::string& q,
    Handler&& handler)
{
    typedef typename std::decay<Handler>::type stored_handler_t;

    // If q is not a correct ip address, do not resolve at all.
    boost::system::error_code ec;
    auto address = boost::asio::ip::address::from_string(q, ec);
    if (ec)
    {
        post(resolver, [ec, h = stored_handler_t(std::forward<Handler>(handler))]() mutable {
            h(ec, iterator(true));
        });
        return;
    }
    boost::asio::ip::tcp::endpoint ep(address, 0);
    async_resolve(resolver, ep, true, std::forward<Handler>(handler));
}

template <typename Handler>
void asio_resolver_service::async_resolve_mx(
    resolver& resolver,
    const std::string&,
    Handler&& handler)
{
    typedef typename std::decay<Handler>::type stored_handler_t;
    post(resolver, [h = stored_handler_t(std::forward<Handler>(handler))]() mutable {
        h(boost::system::errc::make_error_code(boost::system::errc::operation_not_supported),
          iterator());
    });
}

template <typename Handler>
void asio_resolver_service::async_resolve_txt(
    resolver& resolver,
    const std::string&,
    Handler&& handler)
{
    typedef typename std::decay<Handler>::type stored_handler_t;
    post(resolver, [h = stored_handler_t(std::forward<Handler>(handler))]() {
        h(boost::system::errc::make_error_code(boost::system::errc::operation_not_supported),
          iterator());
    });
}

template <typename Query, typename Handler>
void asio_resolver_service::async_resolve(
    resolver& resolver,
    const Query& q,
    bool hostname,
    Handler&& handler)
{
    typedef typename std::decay<Handler>::type stored_handler_t;
    auto cb = [h = stored_handler_t(std::forward<Handler>(handler)),
               hostname](const boost::system::error_code& e, asio_iterator_type it) mutable {
        // XXX check asio invoke is correct
        h(e, iterator(it, hostname));
    };
    if (resolver.strand)
    {
        resolver.asio_resolver->async_resolve(q, resolver.strand->wrap(std::move(cb)));
    }
    else
    {
        resolver.asio_resolver->async_resolve(q, std::move(cb));
    }
}

template <typename Handler>
void asio_resolver_service::post(resolver& resolver, Handler&& handler)
{
    if (resolver.strand)
    {
        resolver.strand->post(std::forward<Handler>(handler));
    }
    else
    {
        io_service.post(std::forward<Handler>(handler));
    }
}

}}}}
