#pragma once

#include <yplatform/net/dns/detail/cache.h>
#include <yplatform/net/dns/resolver_settings.h>
#include <yplatform/log.h>

#include <utility>

namespace yplatform { namespace net { namespace dns { namespace detail {

template <typename ServiceImpl>
class cache_resolver_service
{
    typedef cache_resolver_service<ServiceImpl> this_type;
    typedef cache<typename ServiceImpl::iterator_a> cache_a_t;
    typedef cache<typename ServiceImpl::iterator_aaaa> cache_aaaa_t;
    typedef cache<typename ServiceImpl::iterator_ptr> cache_ptr_t;
    typedef cache<typename ServiceImpl::iterator_mx> cache_mx_t;
    typedef cache<typename ServiceImpl::iterator_txt> cache_txt_t;

    template <typename Cache, typename Handler>
    struct cache_update_handler
    {
        cache_update_handler(
            Cache& cache,
            resolver_service_settings& settings,
            const std::string& name,
            Handler&& handler)
            : cache_(cache)
            , settings_(settings)
            , name_(name)
            , handler_(std::forward<Handler>(handler))
        {
        }

        void operator()(const boost::system::error_code& ec, typename Cache::iterator iter)
        {
            if (!ec)
            {
                cache_.update(name_, iter, settings_.cache_ttl);
            }
            handler_(ec, iter);
        }

        Cache& cache_;
        resolver_service_settings& settings_;
        std::string name_;
        typename std::decay<Handler>::type handler_;
    };

public:
    // Internal data associated with a single resolver instance.
    typedef typename ServiceImpl::resolver resolver;

    cache_resolver_service(boost::asio::io_service& io) : service_impl_(io)
    {
    }

    void setup_service(const resolver_service_settings& settings)
    {
        settings_ = settings;
    }

    void construct(resolver& impl)
    {
        service_impl_.construct(impl);
    }

    void move_construct(resolver& impl, resolver& other_impl)
    {
        service_impl_.move_construct(impl, other_impl);
    }

    void destroy(resolver& impl)
    {
        service_impl_.destroy(impl);
    }

    void shutdown_service()
    {
        service_impl_.shutdown_service();
    }

    // Resolve IPv4 addresses.
    template <typename Handler>
    void async_resolve_a(resolver& impl, const std::string& q, Handler&& handler)
    {
        if (settings_.use_cache())
        {
            async_resolve(
                impl, q, cache_a_, std::forward<Handler>(handler), &ServiceImpl::async_resolve_a);
        }
        else
        {
            service_impl_.async_resolve_a(impl, q, std::forward<Handler>(handler));
        }
    }

    // Resolve IPv6 addresses.
    template <typename Handler>
    void async_resolve_aaaa(resolver& impl, const std::string& q, Handler&& handler)
    {
        if (settings_.use_cache())
        {
            async_resolve(
                impl,
                q,
                cache_aaaa_,
                std::forward<Handler>(handler),
                &ServiceImpl::async_resolve_aaaa);
        }
        else
        {
            service_impl_.async_resolve_aaaa(impl, q, std::forward<Handler>(handler));
        }
    }

    // Back resolve.
    template <typename Handler>
    void async_resolve_ptr(resolver& impl, const std::string& q, Handler&& handler)
    {
        if (settings_.use_cache())
        {
            async_resolve(
                impl,
                q,
                cache_ptr_,
                std::forward<Handler>(handler),
                &ServiceImpl::async_resolve_ptr);
        }
        else
        {
            service_impl_.async_resolve_ptr(impl, q, std::forward<Handler>(handler));
        }
    }

    // Make MX (mail exchanger) records lookup.
    template <typename Handler>
    void async_resolve_mx(resolver& impl, const std::string& q, Handler&& handler)
    {
        if (settings_.use_cache())
        {
            async_resolve(
                impl, q, cache_mx_, std::forward<Handler>(handler), &ServiceImpl::async_resolve_mx);
        }
        else
        {
            service_impl_.async_resolve_mx(impl, q, std::forward<Handler>(handler));
        }
    }

    // Lookup TXT records.
    template <typename Handler>
    void async_resolve_txt(resolver& impl, const std::string& q, Handler&& handler)
    {
        if (settings_.use_cache())
        {
            async_resolve(
                impl,
                q,
                cache_txt_,
                std::forward<Handler>(handler),
                &ServiceImpl::async_resolve_txt);
        }
        else
        {
            service_impl_.async_resolve_txt(impl, q, std::forward<Handler>(handler));
        }
    }

    // cancel current resolve request.
    void cancel(resolver& impl)
    {
        service_impl_.cancel(impl);
    }

private:
    template <typename Cacher, typename Handler>
    void async_resolve(
        resolver& impl,
        const std::string& q,
        Cacher& cacher,
        Handler&& handler,
        void (ServiceImpl::*
                  f)(resolver&, const std::string&, cache_update_handler<Cacher, Handler>&&))
    {
        auto cache_res = cacher.find(q);
        if (cache_res == typename Cacher::iterator())
        {
            (service_impl_.*
             f)(impl,
                q,
                cache_update_handler<Cacher, Handler>(
                    cacher, settings_, q, std::forward<Handler>(handler)));
        }
        else
        {
            handler(boost::system::error_code(), cache_res);
        }
    }

    resolver_service_settings settings_;
    ServiceImpl service_impl_;

    cache_a_t cache_a_;
    cache_aaaa_t cache_aaaa_;
    cache_ptr_t cache_ptr_;
    cache_mx_t cache_mx_;
    cache_txt_t cache_txt_;
};

}}}}